#' Plot Model Assessment for S-FFA
#'
#' Creates a quantile–quantile plot comparing observed annual maximum series
#' data to quantile estimates from a fitted parametric model. The 1:1 line is drawn in 
#' black and the parametric model estimates are plotted as semi‐transparent red points.
#'
#' @param results List; model assessment results generated by [model_assessment()].
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return \code{ggplot}; a plot containing:
#' - A black line representing a model with no deviation from the empirical quantiles.
#' - Red points denoting the estimated quantiles against the empirical quantiles.
#'
#' @examples
#' # Initialize example data
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#'
#' # Evaluate model diagnostics
#' results <- model_assessment(data, "NOR", "L-moments")
#'
#' # Generate a model assessment plot
#' plot_sffa_assessment(results)
#'
#' @import ggplot2
#' @export

plot_sffa_assessment <- function(results, ...) {

	# Create a dataframe for the plot
	df <- data.frame(x = results$q_theoretical, y = results$q_empirical)

	# Capture optional arguments
	args <- list(...)

    # Set default values, overriding if necessary
    title <- args$title %||% "S-FFA Model Assessment"
    xlabel <- args$xlabel %||% expression(Theoretical ~ Quantiles ~ m^3/s)
	ylabel <- args$ylabel %||% expression(Empirical ~ Quantiles ~ m^3/s)

	# Generate the plot
	p1 <- ggplot(data = df) +
		geom_line(
			aes(x = .data$x, y = .data$x),
			linewidth = 1.1
		) + 
		geom_point(
			aes(x = .data$x, y = .data$y), 
			color = "red",
			size = 3,
			alpha = 0.5
		) + 
		labs(
			title = title,
			x = xlabel,
			y = ylabel
		)

	# Add theme and scales and return
	add_theme(add_scales(p1))

}
