% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reg.R
\name{reg}
\alias{reg}
\title{Create 'reg' (regularisation) object}
\usage{
reg(reg_type = "l1", a = NULL, lower = -Inf, upper = Inf,
  lambda_factor = 1, exclude = NULL, penalty_factor = NULL,
  contexts = NULL, scales = NULL, fixed = FALSE, screen = NULL,
  exact_gradient = NULL, step_max = 200,
  step_screen = ceiling(step_max/10), step_cycle = step_screen,
  backtrack_max = 50, tol = 1e-05, tau_init = 0.1, tau_min = 1e-10,
  tau_scale = 0.5)
}
\arguments{
\item{reg_type}{Character string determining the regularisation. Must be one of: "l1" (default), "l2", "elnet", "scad", "mcp" or "none". See details.}

\item{a}{Numeric value of tuning parameter in \code{elnet} (must be between 0 and 1, default is 0.5), \code{scad} (must be larger than 2, default = 3.7) or \code{mcp} (must be larger than 1, default = 3).}

\item{lower}{Either numeric vector of length 1 or p, of lower limit(s) for parameters, must be smaller or equal to \code{upper}.}

\item{upper}{Either numeric vector of length 1 or p, of upper limit(s) for parameters, must be larger or equal to \code{lower}.}

\item{lambda_factor}{Non-negative numeric value which will be multiplied with the tuning parameter in \code{\link{opt}}.}

\item{exclude}{A vector indices to exclude from model (this is how to specify an infinite penalty factor). Default is none.}

\item{penalty_factor}{A non-negative vector (p) of individual penalty weights. Defaults to 1 for each parameter. Will always be rescaled so its mean is 1.}

\item{contexts}{A non-negative matrix (pxs) specifying design on context-level, see details. Defaults to a matrix of ones.}

\item{scales}{A positive vector (p) of scales, see details. Defaults to a vector of ones. Will be rescaled to mean to 1.}

\item{fixed}{Logical indicating if this parameter is fixed or should be optimised.}

\item{screen}{Logical indicating if a faster optimisation relying on screening should be adapted. If \code{NULL}, it is set to \code{TRUE} iff p > 20.}

\item{exact_gradient}{Logical indicating if exact gradient should be used. If \code{NULL}, it is set to \code{TRUE} iff p > 20.}

\item{step_max}{Positive integer giving the maximal number of steps in optimisation procedure, per lambda.}

\item{step_screen}{Positive integer giving the number of steps between screenings (defaults to \code{ceiling(step_max / 50)}).}

\item{step_cycle}{Positive integer giving the number of steps per optimisation cycle, defaults to \code{step_screen}.}

\item{backtrack_max}{Positive integer giving the maximal number of backtracking steps taken in each optimisation step.}

\item{tol}{Positive numeric tolerance level used for parameter space.}

\item{tau_init}{Positive initial step length for backtracking.}

\item{tau_min}{Positive numeric giving minimal value of step length in backtracking.}

\item{tau_scale}{Scaling parameter of step length, must be in (0,1).}
}
\value{
An object with S3 class "reg".
}
\description{
This function creates an object of class \code{reg}, which holds regularisation type, tuning parameter scales, penalty factors and control list for optimisation. This is basically the control panel for the optimisation in \code{\link{rodeo}} and \code{\link{aim}}.
}
\details{
\subsection{Data format}{
The data (\code{y} in \code{\link{opt}}) is assumed generated from s different contexts. A new context begins whenever the time (the first column of \code{y} in \code{\link{opt}}) decreases. Hence s - 1 is the number of decreases in the time points.

Each context has its own initial condition and parameter vector specified through \code{contexts} in \code{\link{reg}}. More precisely, the effective parameter in context l is the element-wise product of a baseline parameter (scaled by \code{scales} in \code{\link{reg}}) and the l'th column of \code{contexts}. This enables the user to pre-specify different scales for each parameter, as well as different scales for the same parameter across contexts.

The default setup sets \code{contexts} to a matrix of ones. The following are examples of case-specific modifications for the \code{\link{mak}} ODE class:
If reaction \code{j} does not take place in context \code{l} then set \eqn{contexts_{j,l} = 0}.
If reaction \code{j} has a \eqn{50\%} increase in rate in context \code{l} then set \eqn{contexts_{j,l} = 1.5}.
If reaction \code{j} has independent rates in contexts \code{l} and \code{l}', then write that reaction twice in \code{\link{mak}}-object (with \code{j}' denoting its second appearance) and set \eqn{contexts_{j,l'} = 0} and \eqn{contexts_{j',l} = 0}.
}
\subsection{Loss function}{
The loss function optimised in \code{\link{rodeo}} is:
\deqn{RSS/(2 * (n - s)) + \lambda*\sum_{parameter \ argument}\lambda_{factor}*\sum_{j=1}^p v_jpen(param_j)}
where \eqn{\lambda} belongs to the \code{lambda}-sequence in \code{\link{opt}}-object and v is \code{penalty_factor} in \code{\link{reg}}. Moreover, the residual sum of squares, RSS, is given as:
\deqn{RSS = \sum^{n}_{i=1}||(y(t_i) - x(t_i, {x_0}_l, context_l * param))*\sqrt{w(t_i)}||_2^2}
where \code{param} has been (internally) scaled with \code{scales}, and \eqn{w(t_i)} and \eqn{y(t_i)} refers to the i'th row of \code{weights} and \code{y} in \code{\link{opt}} (\code{y} with first column removed), respectively. The solution to the ODE system is the x()-function. The subscript 'l' refers to the context, i.e., the columns of \code{contexts}  and \code{x0} in \code{rodeo}-functions (\code{x0} is the initial state of the system at the first time point after a decrease in the time-vector). All products are Hadamard products.
}
\subsection{Regularisation}{
The type of regularisation is chosen via \code{reg_type} in \code{\link{reg}}:
\describe{
\item{\code{l1}:}{Least Absolute Shrinkage and Selection Operator (Lasso) penalty. The penalty is the absolute value: \eqn{pen(param_j)=|param_j|}}
\item{\code{l2}:}{Ridge penalty. The penalty is the squaring: \eqn{pen(param_j)=param_j^2/2}}
\item{\code{elnet}:}{Elastic Net. A convex combination of \code{l1} and \code{l2} penalties: \eqn{pen(param_j)=(1-a)param_j^2/2+a|param_j|}, 0<=a<=1. Note if a = 0 or a = 1, then the penalty is automatically reduced to "l2" and "l1" respectively.}
\item{\code{scad}:}{Smoothly Clipped Absolute Deviation penalty: \deqn{pen(param_j)=\int^{param_j}_{0}{min( max((a\lambda-|param|)/(\lambda(a-1)), 0), 1) dparam}, a > 2.}}
\item{\code{mcp}:}{Maximum Concave Penalty penalty: \deqn{pen(param_j)=\int^{param_j}_{0}{max(1 - |param|/(\lambda a), 0) dparam}, a > 1.}}
\item{\code{none}:}{No penalty. Not recommended for large systems. This overwrites both user-supplied and automatically generated lambda-sequences.}
}
}
\subsection{Optimisation}{
 A proximal-gradient type of optimisation is employed. The step length is denoted \eqn{\tau}.

 The flag \code{screen} (in \code{\link{reg}}) is passed onto the optimisation, it simply tells the optimisation to focus entirely on optimising a subset of the parameters, which where selected due to large gradients. At most \code{step_screen} (in \code{\link{reg}}) steps are taken before a re-evaluation of the screened subset takes place.

The convergence criteria is \eqn{\Delta \eta < 10^3 * \max(|(\Delta param \neq 0)| + |(\Delta x0 \neq 0)|, 1)} AND \eqn{\Delta loss / \Delta \eta < tol_l}, where
 \deqn{\Delta \eta = ||\Delta param||/tol_{param} + ||\Delta x0|| / tol_{x_0}}

}
\subsection{Tuning parameter}{
The lambda sequence can either be fully customised through \code{lambda} in \code{\link{opt}} or automatically generated. In the former case, a monotonic \code{lambda}-sequence is highly recommended. Throughout all optimisations, each optimisation step re-uses the old optimum, when sweeping through the \code{lambda}-sequence.

If \code{lambda} is \code{NULL}, an automatically generated sequence is used. A maximal value of lambda (the smallest at which 0 is a optimum in the rate parameter) is calculated and log-equidistant sequence of length \code{nlambda} is generated, with the ratio between the smallest and largest lambda value being \code{lambda.min.ratio}. Note: when the \code{opt}-object is passed to \code{\link{rodeo.ode}}, one may want to initialise the optimisation at a non-zero parameter and run the optimisation on the reversed lambda sequence. This is indicated by setting \code{decrease.lambda} = FALSE. If, however, the \code{opt}-object is passed to \code{\link{aim}}, \code{\link{glmnet}} ultimately decides on the lambda sequence, and may cut it short.
}
\subsection{Gradient evaluations}{
Two gradient evaluations are implemented: exact and approximate. The computational time of exact gradient is generally longer than that of approximate gradient, but its relative magnitude depends on the \code{solver} in the \code{ode}-object passed to \code{\link{rodeo}}.
}
}
\examples{
# Use 'reg' object to specify regularisation when creating 'ode' objects

# Example: power law kinetics with SCAD penalty on rate parameter
A <- matrix(
c(1, 1, 0, 0,
  0, 0, 1, 0,
  0, 0, 1, 0), ncol = 4, byrow = TRUE)
p <- plk(A, r = reg("scad"))

# Example: power law kinetics as above, with lower bound of -1
p <- plk(A, r = reg("scad", lower = -1))

# Example: rational mass action kinetics with
# MCP penalty on first parameter argument and l2 on second
B <- matrix(
c(0, 0, 1, 0,
  1, 1, 0, 0,
  1, 0, 0, 1), ncol = 4, byrow = TRUE)
rmak <- ratmak(A, B, r1 = reg("mcp"), r2 = reg("l2"))

# Example: mass action kinetics with
# no penalty on rate parameter and l2 on initial state
m <- mak(A, B, r = reg("none"), rx0 = reg("l2"))

}
\seealso{
aim, rodeo, rodeo.aim, rodeo.ode
}
