% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dists.R
\name{dodgr_dists}
\alias{dodgr_dists}
\title{dodgr_dists}
\usage{
dodgr_dists(graph, from, to, wt_profile = "bicycle", expand = 0,
  heap = "BHeap", parallel = TRUE, quiet = TRUE)
}
\arguments{
\item{graph}{\code{data.frame} or equivalent object representing the network
graph (see Details)}

\item{from}{Vector or matrix of points **from** which route distances are to
be calculated (see Details)}

\item{to}{Vector or matrix of points **to** which route distances are to be
calculated (see Details)}

\item{wt_profile}{Name of weighting profile for street networks (one of foot,
horse, wheelchair, bicycle, moped, motorcycle, motorcar, goods, hgv, psv).}

\item{expand}{Only when \code{graph} not given, the multiplicative factor by
which to expand the street network surrounding the points defined by
\code{from} and/or \code{to}.}

\item{heap}{Type of heap to use in priority queue. Options include
Fibonacci Heap (default; \code{FHeap}), Binary Heap (\code{BHeap}),
\code{Radix}, Trinomial Heap (\code{TriHeap}), Extended Trinomial Heap
(\code{TriHeapExt}, and 2-3 Heap (\code{Heap23}).}

\item{parallel}{If \code{TRUE}, perform routing calculation in parallel (see
details)}

\item{quiet}{If \code{FALSE}, display progress messages on screen.}
}
\value{
square matrix of distances between nodes
}
\description{
Calculate matrix of pair-wise distances between points.
}
\note{
\code{graph} must minimally contain three columns of \code{from},
\code{to}, \code{dist}. If an additional column named \code{weight} or
\code{wt} is present, shortest paths are calculated according to values
specified in that column; otherwise according to \code{dist} values. Either
way, final distances between \code{from} and \code{to} points are calculated
according to values of \code{dist}. That is, paths between any pair of points
will be calculated according to the minimal total sum of \code{weight}
values (if present), while reported distances will be total sums of
\code{dist} values.

The \code{from} and \code{to} columns of \code{graph} may be either single
columns of numeric or character values specifying the numbers or names of
graph vertices, or combinations to two columns specifying geographical
(longitude and latitude) coordinates. In the latter case, almost any sensible
combination of names will be accepted (for example, \code{fromx, fromy},
\code{from_x, from_y}, or \code{fr_lat, fr_lon}.)

\code{from} and \code{to} values can be either two-column matrices of
equivalent of longitude and latitude coordinates, or else single columns
precisely matching node numbers or names given in \code{graph$from} or
\code{graph$to}. If \code{to} is missing, pairwise distances are calculated
between all points specified in \code{from}. If neither \code{from} nor
\code{to} are specified, pairwise distances are calculated between all nodes
in \code{graph}.

Calculations in parallel (\code{parallel = TRUE}) ought very generally be
advantageous. For small graphs, Calculating distances in parallel is likely
to offer relatively little gain in speed, but increases from parallel
computation will generally markedly increase with increasing graph sizes.
}
\examples{
# A simple graph
graph <- data.frame (from = c ("A", "B", "B", "B", "C", "C", "D", "D"),
                     to = c ("B", "A", "C", "D", "B", "D", "C", "A"),
                     d = c (1, 2, 1, 3, 2, 1, 2, 1))
dodgr_dists (graph)

# A larger example from the included \\code{\\link{hampi}} data.
graph <- weight_streetnet (hampi)
from <- sample (graph$from_id, size = 100)
to <- sample (graph$to_id, size = 50)
d <- dodgr_dists (graph, from = from, to = to)
# d is a 100-by-50 matrix of distances between \\code{from} and \\code{to}

\dontrun{
# a more complex street network example, thanks to @chrijo; see
# https://github.com/ATFutures/dodgr/issues/47

xy <- rbind (c (7.005994, 51.45774), # limbeckerplatz 1 essen germany
             c (7.012874, 51.45041)) # hauptbahnhof essen germany
xy <- data.frame (lon = xy [, 1], lat = xy [, 2])
essen <- dodgr_streetnet (pts = xy, expand = 0.2, quiet = FALSE)
graph <- weight_streetnet (essen, wt_profile = "foot")
d <- dodgr_dists (graph, from = xy, to = xy)
# First reason why this does not work is because the graph has multiple,
# disconnected components.
table (graph$component)
# reduce to largest connected component, which is always number 1
graph <- graph [which (graph$component == 1), ]
d <- dodgr_dists (graph, from = xy, to = xy)
# should work, but even then note that
table (essen$level)
# There are parts of the network on different building levels (because of
# shopping malls and the like). These may or may not be connected, so it may be
# necessary to filter out particular levels
levs <- paste0 (essen$level) # because sf data are factors
index <- which (! (levs == "-1" | levs == "1")) # for example
library (sf) # needed for following sub-select operation
essen <- essen [index, ]
graph <- weight_streetnet (essen, wt_profile = "foot")
d <- dodgr_dists (graph, from = xy, to = xy)
}
}
