% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/zoo_permute.R
\name{zoo_permute}
\alias{zoo_permute}
\title{Random or Restricted Permutation of Zoo Time Series}
\usage{
zoo_permute(
  x = NULL,
  repetitions = 1L,
  permutation = "restricted_by_row",
  block_size = NULL,
  seed = 1L
)
}
\arguments{
\item{x}{(required, zoo object) zoo time series. Default: NULL}

\item{repetitions}{(optional, integer) number of permutations to compute. Large numbers may compromise your R session. Default: 1}

\item{permutation}{(optional, character string) permutation method. Valid values are listed below from higher to lower induced randomness:
\itemize{
\item "free": unrestricted re-shuffling of individual cases across rows and columns. Ignores block_size.
\item "free_by_row": unrestricted re-shuffling of complete rows. Ignores block size.
\item "restricted": restricted shuffling across rows and columns within blocks of rows.
\item "restricted_by_row": restricted re-shuffling of rows within blocks.
}}

\item{block_size}{(optional, integer) Block size in number of rows for restricted permutations. Only relevant when permutation methods are "restricted" or "restricted_by_row". A block of size \code{n} indicates that the original data is pre-divided into blocks of such size, and a given row can only be permuted within their original block. If NULL, defaults to the rounded one tenth of the number of rows in \code{x}. Default: NULL.}

\item{seed}{(optional, integer) initial random seed to use during permutations. Default: 1}
}
\value{
Time Series List
}
\description{
Fast permutation of zoo time series for null model testing using a fast and efficient C++ implementations of different restricted and free permutation methods.

The available permutation methods are:

\itemize{
\item "free" (see \code{\link[=permute_free_cpp]{permute_free_cpp()}}): Unrestricted and independent re-shuffling of individual cases across rows and columns. Individual values are relocated to a new row and column within the dimensions of the original matrix.
\item "free_by_row" (see \code{\link[=permute_free_by_row_cpp]{permute_free_by_row_cpp()}}): Unrestricted re-shuffling of complete rows. Each individual row is given a new random row number, and the data  matrix is re-ordered accordingly.
\item "restricted" (see \code{\link[=permute_restricted_cpp]{permute_restricted_cpp()}}): Data re-shuffling across rows and columns is restricted to blocks of contiguous rows. The algorithm divides the data matrix into a set of blocks of contiguous rows, and individual cases are then assigned to a new row and column within their original block.
\item "restricted_by_row" (see \code{\link[=permute_restricted_by_row_cpp]{permute_restricted_by_row_cpp()}}): Re-shuffling of complete rows is restricted to blocks of contiguous rows. The algorithm divides the data matrix into a set of blocks of contiguous rows, each individual row is given a new random row number within its original block, and the block is reordered accordingly to generate the permuted output.
}

This function supports a parallelization setup via \code{\link[future:plan]{future::plan()}}, and progress bars provided by the package \href{https://CRAN.R-project.org/package=progressr}{progressr}.
}
\examples{

#simulate zoo time series
x <- zoo_simulate(cols = 2)

if(interactive()){
  zoo_plot(x)
}

#free
x_free <- zoo_permute(
  x = x,
  permutation = "free",
  repetitions = 2
)

if(interactive()){
  tsl_plot(
    tsl = x_free,
    guide = FALSE
    )
}

#free by row
x_free_by_row <- zoo_permute(
  x = x,
  permutation = "free_by_row",
  repetitions = 2
)

if(interactive()){
  tsl_plot(
    tsl = x_free_by_row,
    guide = FALSE
  )
}

#restricted
x_restricted <- zoo_permute(
  x = x,
  permutation = "restricted",
  repetitions = 2
)

if(interactive()){
  tsl_plot(
    tsl = x_restricted,
    guide = FALSE
  )
}

#restricted by row
x_restricted_by_row <- zoo_permute(
  x = x,
  permutation = "restricted_by_row",
  repetitions = 2
)

if(interactive()){
  tsl_plot(
    tsl = x_restricted_by_row,
    guide = FALSE
  )
}

}
\seealso{
Other zoo_functions: 
\code{\link{zoo_aggregate}()},
\code{\link{zoo_name_clean}()},
\code{\link{zoo_name_get}()},
\code{\link{zoo_name_set}()},
\code{\link{zoo_plot}()},
\code{\link{zoo_resample}()},
\code{\link{zoo_smooth_exponential}()},
\code{\link{zoo_smooth_window}()},
\code{\link{zoo_time}()},
\code{\link{zoo_to_tsl}()},
\code{\link{zoo_vector_to_matrix}()}
}
\concept{zoo_functions}
