% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statistics.R
\name{get_sim_lim}
\alias{get_sim_lim}
\title{Similarity limit}
\usage{
get_sim_lim(mtad, lhs)
}
\arguments{
\item{mtad}{A numeric value specifying the \dQuote{maximum tolerable average
difference} (MTAD) of the profiles of two formulations at all time points
(in \%). The default value is \code{10}. It determines the size of the
similarity limit \eqn{\bm{d}_g}{d_g}.}

\item{lhs}{A list of the estimates of Hotelling's two-sample \eqn{T^2}
statistic for small samples as returned by the function
\code{\link{get_hotellings}()}.}
}
\value{
A vector containing the following information is returned:
\item{DM}{The Mahalanobis distance of the samples.}
\item{df1}{Degrees of freedom (number of variables or time points).}
\item{df2}{Degrees of freedom (number of rows - number of variables - 1).}
\item{alpha}{The provided significance level.}
\item{K}{Scaling factor for \eqn{F} to account for the distribution of the
  \eqn{T^2} statistic.}
\item{k}{Scaling factor for the squared Mahalanobis distance to obtain
  the \eqn{T^2} statistic.}
\item{T2}{Hotelling's \eqn{T^2} statistic (\eqn{F}-distributed).}
\item{F}{Observed \eqn{F} value.}
\item{ncp.Hoffelder}{Non-centrality parameter for calculation of the \eqn{F}
  statistic (\eqn{T^2} test procedure).}
\item{F.crit}{Critical \eqn{F} value (Tsong's procedure).}
\item{F.crit.Hoffelder}{Critical \eqn{F} value (\eqn{T^2} test procedure).}
\item{p.F}{The \eqn{p} value for the Hotelling's \eqn{T^2} test statistic.}
\item{p.F.Hoffelder}{The \eqn{p} value for the Hotelling's \eqn{T^2}
  statistic based on the non-central \eqn{F} distribution.}
\item{MTAD}{Specified \dQuote{maximum tolerable average difference} (MTAD)
  of the profiles of two formulations at each individual time point (in \%).}
\item{Sim.Limit}{Critical Mahalanobis distance or similarity limit
  (Tsong's procedure).}
}
\description{
The function \code{get_sim_lim()} estimates a similarity limit in terms of
the \dQuote{Multivariate Statistical Distance} (MSD).
}
\details{
Details about the estimation of similarity limits in terms of
the \dQuote{Multivariate Statistical Distance} (MSD) are explained in
the corresponding section below.
}
\section{Similarity limits in terms of MSD}{

For the calculation of the \dQuote{Multivariate Statistical Distance} (MSD),
the procedure proposed by Tsong et al. (1996) can be considered as
well-accepted method that is actually recommended by the FDA. According
to this method, a multivariate statistical distance, called Mahalanobis
distance, is used to measure the difference between two multivariate means.
This distance measure is calculated as

\deqn{D_M = \sqrt{ \left( \bm{x_T} - \bm{x_R} \right)^{\top}
  \bm{S}_{pooled}^{-1} \left( \bm{x_T} - \bm{x_R} \right)} ,}{%
  D_M = sqrt((x_T - x_R)^{\top} S_{pooled}^{-1} (x_T - x_R)) ,}

where \eqn{\bm{S}_{pooled} = \frac{\left( \bm{S}_T + \bm{S}_R \right)}{2}}{%
S_{pooled} = (S_T + S_R) / 2} is the sample variance-covariance matrix
pooled across the batches, \eqn{\bm{x}_T}{x_T} and \eqn{\bm{x}_R}{x_R} are
the vectors of the sample means for the test (\eqn{T}) and reference
(\eqn{R}) profiles, and \eqn{\bm{S}_T}{S_T} and \eqn{\bm{S}_R}{x_R} are the
variance-covariance matrices of the test and reference profiles.

In order to determine the similarity limits in terms of the MSD, i.e. the
Mahalanobis distance between the two multivariate means of the dissolution
profiles of the formulations to be compared, Tsong et al. (1996) proposed
using the equation

\deqn{D_M^{max} = \sqrt{ \bm{d}_g^{\top} \bm{S}_{pooled}^{-1} \bm{d}_g} ,}{%
  D_M^{max} = sqrt(d_g^{\top} S_{pooled}^{-1} d_g) ,}

where \eqn{\bm{d}_g}{d_g} is a \eqn{1 \times p}{1 x p} vector with all
\eqn{p} elements equal to an empirically defined limit \eqn{\bm{d}_g}{d_g},
e.g., \eqn{15}\%, for the maximum tolerable difference at all time points,
and \eqn{p} is the number of sampling points. By assuming that the data
follow a multivariate normal distribution, the 90\% confidence region
(\eqn{CR}) bounds for the true difference between the mean vectors,
\eqn{\bm{\mu}_T - \bm{\mu}_R}{\mu_T - \mu_R}, can be computed for the
resultant vector \eqn{\bm{\mu}}{\mu} to satisfy the following condition:

\deqn{\bm{CR} = K \left( \bm{\mu} - \left( \bm{x_T} - \bm{x_R} \right)
  \right)^{\top} \bm{S}_{pooled}^{-1} \left( \bm{\mu} - \left( \bm{x_T} -
  \bm{x_R} \right) \right) \leq F_{p, n_T + n_R - p - 1, 0.9} ,}{%
  CR = sqrt((\mu - (x_T - x_R))^{\top} S_{pooled}^{-1} (\mu - (x_T - x_R)))
  \leq F_{p, n_T + n_R - p - 1, 0.9} ,}

where \eqn{K} is the scaling factor that is calculated as

\deqn{K = \frac{n_T n_R}{n_T + n_R} \cdot \frac{n_T + n_R - p - 1}{
  \left( n_T + n_R - 2 \right) \cdot p} ,}{%
  (n_T n_R) / (n_T + n_R) * (n_T + n_R - p - 1) / ((n_T + n_R - 2) p) ,}

and \eqn{F_{p, n_T + n_R - p - 1, 0.9}} is the \eqn{90^{th}} percentile of
the \eqn{F} distribution with degrees of freedom \eqn{p} and
\eqn{n_T + n_R - p - 1}. It is obvious that \eqn{(n_T + n_R)} must be greater
than \eqn{(p + 1)}. The formula for \eqn{CR} gives a \eqn{p}-variate 90\%
confidence region for the possible true differences. \cr
}

\section{T2 test for equivalence}{

Based on the distance measure for profile comparison that was suggested by
Tsong et al. (1996), i.e. the Mahalanobis distance, Hoffelder (2016) proposed
a statistical equivalence procedure for that distance, the so-called
\eqn{T^2} test for equivalence (T2EQ). It is used to demonstrate that the
Mahalanobis distance between reference and test group dissolution profiles
is smaller than the \dQuote{Equivalence Margin} (EM). Decision in favour of
equivalence is taken if the \eqn{p} value of this test statistic is smaller
than the pre-specified significance level \eqn{\alpha}, i.e. if
\eqn{p < \alpha}. The \eqn{p} value is calculated by aid of the formula

\deqn{p = F_{p, n_T + n_R - p - 1, ncp, \alpha}
  \frac{n_T + n_R - p - 1}{(n_T + n_R - 2) p} T^2 ,}{%
  p = F_{p, n_T + n_R - p - 1, ncp, \alpha}
  (n_T + n_R - p - 1) / ((n_T + n_R - 2) p) T^2 ,}

where \eqn{\alpha} is the significance level and \eqn{ncp} is the so-called
\dQuote{\emph{non-centrality parameter}} that is calculated by

\deqn{\frac{n_T n_R}{n_T + n_R} \left( D_M^{max} \right)^2 .}{%
  (n_T n_R) / (n_T + n_R) (D_M^{max})^2 .}

The test statistic being used is Hotelling's \eqn{T^2} that is given as

\deqn{T^2 = \frac{n_T n_R}{n_T + n_R} \left( \bm{x_T} - \bm{x_R}
  \right)^{\top} \bm{S}_{pooled}^{-1} \left( \bm{x_T} - \bm{x_R} \right) .}{%
  (n_T n_R) / (n_T + n_R) * (x_T - x_R)^{\top} S_{pooled}^{-1} (x_T - x_R) .}

As mentioned elsewhere, \eqn{\bm{d}_g}{d_g} is a \eqn{1 \times p}{1 x p}
vector with all \eqn{p} elements equal to an empirically defined limit
\eqn{d_g}. Thus, the components of the vector \eqn{\bm{d}_g}{d_g} can be
interpreted as upper bound for a kind of \dQuote{\emph{average}} allowed
difference between test and reference profiles, the \dQuote{\emph{global
similarity limit}}. Since the EMA requires that \dQuote{similarity acceptance
limits should be pre-defined and justified and not be greater than a 10\%
difference}, it is recommended to use 10\%, not 15\% as proposed by Tsong
et al. (1996), for the maximum tolerable difference at all time points.
}

\examples{
# Dissolution data of one reference batch and one test batch of n = 6
# tablets each:
str(dip1)

# 'data.frame':	12 obs. of  10 variables:
# $ type  : Factor w/ 2 levels "R","T": 1 1 1 1 1 1 2 2 2 2 ...
# $ tablet: Factor w/ 6 levels "1","2","3","4",..: 1 2 3 4 5 6 1 2 3 4 ...
# $ t.5   : num  42.1 44.2 45.6 48.5 50.5 ...
# $ t.10  : num  59.9 60.2 55.8 60.4 61.8 ...
# $ t.15  : num  65.6 67.2 65.6 66.5 69.1 ...
# $ t.20  : num  71.8 70.8 70.5 73.1 72.8 ...
# $ t.30  : num  77.8 76.1 76.9 78.5 79 ...
# $ t.60  : num  85.7 83.3 83.9 85 86.9 ...
# $ t.90  : num  93.1 88 86.8 88 89.7 ...
# $ t.120 : num  94.2 89.6 90.1 93.4 90.8 ...

# Estimation of the parameters for Hotelling's two-sample T2 statistic
# (for small samples)
hs <- get_hotellings(m1 = as.matrix(dip1[dip1$type == "R", c("t.15", "t.90")]),
                    m2 = as.matrix(dip1[dip1$type == "T", c("t.15", "t.90")]),
                    signif = 0.1)

# Estimation of the similarity limit in terms of the "Multivariate Statistical
# Distance" (MSD)for a "maximum tolerable average difference" (mtad) of 10
res <- get_sim_lim(mtad = 15, hs)

# Expected results in res
#            DM              df1              df2            alpha
#  1.044045e+01     2.000000e+00     9.000000e+00     1.000000e-01
#             K                k               T2                F
#  1.350000e+00     3.000000e+00     3.270089e+02     1.471540e+02
# ncp.Hoffelder           F.crit F.crit.Hoffelder              p.F
#  2.782556e+02     3.006452e+00     8.357064e+01     1.335407e-07
# p.F.Hoffelder             MTAD        Sim.Limit
#  4.822832e-01     1.500000e+01     9.630777e+00
}
\references{
Tsong, Y., Hammerstrom, T., Sathe, P.M., and Shah, V.P. Statistical
assessment of mean differences between two dissolution data sets.
\emph{Drug Inf J}. 1996; \strong{30}: 1105-1112.\cr
\doi{10.1177/009286159603000427}

Wellek S. (2010) \emph{Testing statistical hypotheses of equivalence and
noninferiority} (2nd ed.). Chapman & Hall/CRC, Boca Raton.\cr
\doi{10.1201/EBK1439808184}

Hoffelder, T. Highly variable dissolution profiles. Comparison of
\eqn{T^2}-test for equivalence and \eqn{f_2} based methods. \emph{Pharm Ind}.
2016; \strong{78}(4): 587-592.\cr
\url{https://www.ecv.de/suse_item.php?suseId=Z|pi|8430}
}
\seealso{
\code{\link{mimcr}}, \code{\link{get_hotellings}}.
}
