% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimNLR.R
\name{estimNLR}
\alias{estimNLR}
\alias{logLik.estimNLR}
\alias{coef.estimNLR}
\alias{fitted.estimNLR}
\alias{residuals.estimNLR}
\alias{print.estimNLR}
\alias{vcov.estimNLR}
\title{Non-linear regression DIF models estimation.}
\usage{
estimNLR(y, match, group, formula, method, lower, upper, start)

\method{logLik}{estimNLR}(object, ...)

\method{coef}{estimNLR}(object, ...)

\method{fitted}{estimNLR}(object, ...)

\method{residuals}{estimNLR}(object, ...)

\method{print}{estimNLR}(x, ...)

\method{vcov}{estimNLR}(object, sandwich = FALSE, ...)
}
\arguments{
\item{y}{numeric: a binary vector of responses (\code{"1"} correct,
\code{"0"} incorrect).}

\item{match}{numeric: a numeric vector describing the matching criterion.}

\item{group}{numeric: a binary vector of a group membership (\code{"0"}
for the reference group, \code{"1"} for the focal group).}

\item{formula}{formula: specification of the model. It can be obtained by the
\code{formulaNLR()} function.}

\item{method}{character: an estimation method to be applied. The options are
\code{"nls"} for non-linear least squares (default), \code{"mle"} for the
maximum likelihood method using the \code{"L-BFGS-B"} algorithm with
constraints, \code{"em"} for the maximum likelihood estimation with the EM
algorithm, \code{"plf"} for the maximum likelihood estimation with the
algorithm based on parametric link function, and \code{"irls"} for the maximum
likelihood estimation with the iteratively reweighted least squares algorithm
(available for the \code{"2PL"} model only). See \strong{Details}.}

\item{lower}{numeric: lower bounds for item parameters of the model specified
in the \code{formula}.}

\item{upper}{numeric: upper bounds for item parameters of the model specified
in the \code{formula}.}

\item{start}{numeric: initial values of item parameters. They can be obtained
by the \code{startNLR()} function.}

\item{object}{an object of the \code{"estimNLR"} class.}

\item{...}{other generic parameters for S3 methods.}

\item{x}{an object of the \code{"estimNLR"} class.}

\item{sandwich}{logical: should the sandwich estimator be applied for
computation of the covariance matrix of item parameters when using
\code{method = "nls"}? (the default is \code{FALSE}).}
}
\description{
Estimates parameters of non-linear regression models for DIF detection using
either non-linear least squares or maximum likelihood method with various
algorithms.
}
\details{
The function offers either the non-linear least squares estimation via the
\code{\link[stats]{nls}} function (Drabinova & Martinkova, 2017; Hladka &
Martinkova, 2020), the maximum likelihood method with the \code{"L-BFGS-B"}
algorithm with constraints via the \code{\link[stats]{optim}} function (Hladka &
Martinkova, 2020), the maximum likelihood method with the EM algorithm (Hladka,
Martinkova, & Brabec, 2025), the maximum likelihood method with the algorithm
based on parametric link function (PLF; Hladka, Martinkova, & Brabec, 2025), or
the maximum likelihood method with the iteratively reweighted least squares
algorithm via the \code{\link[stats]{glm}} function.
}
\examples{
# loading data
data(GMAT)
y <- GMAT[, 1] # item 1
match <- scale(rowSums(GMAT[, 1:20])) # standardized total score
group <- GMAT[, "group"] # group membership variable

# formula for 3PL model with the same guessing for both groups,
# IRT parameterization
M <- formulaNLR(model = "3PLcg", type = "both", parameterization = "irt")

# starting values for 3PL model with the same guessing for item 1
start <- startNLR(GMAT[, 1:20], group, model = "3PLcg", parameterization = "irt")
start <- start[[1]][M$M1$parameters]

# nonlinear least squares
(fit_nls <- estimNLR(
  y = y, match = match, group = group,
  formula = M$M1$formula, method = "nls",
  lower = M$M1$lower, upper = M$M1$upper, start = start
))

coef(fit_nls)
logLik(fit_nls)
vcov(fit_nls)
vcov(fit_nls, sandwich = TRUE)
fitted(fit_nls)
residuals(fit_nls)

# maximum likelihood method
(fit_mle <- estimNLR(
  y = y, match = match, group = group,
  formula = M$M1$formula, method = "mle",
  lower = M$M1$lower, upper = M$M1$upper, start = start
))

coef(fit_mle)
logLik(fit_mle)
vcov(fit_mle)
fitted(fit_mle)
residuals(fit_mle)

# formula for 3PL model with the same guessing for both groups
# intercept-slope parameterization
M <- formulaNLR(model = "3PLcg", type = "both", parameterization = "is")

# starting values for 3PL model with the same guessing for item 1,
start <- startNLR(GMAT[, 1:20], group, model = "3PLcg", parameterization = "is")
start <- start[[1]][M$M1$parameters]

# EM algorithm
(fit_em <- estimNLR(
  y = y, match = match, group = group,
  formula = M$M1$formula, method = "em",
  lower = M$M1$lower, upper = M$M1$upper, start = start
))

coef(fit_em)
logLik(fit_em)
vcov(fit_em)
fitted(fit_em)
residuals(fit_em)

# PLF algorithm
(fit_plf <- estimNLR(
  y = y, match = match, group = group,
  formula = M$M1$formula, method = "plf",
  lower = M$M1$lower, upper = M$M1$upper, start = start
))

coef(fit_plf)
logLik(fit_plf)
vcov(fit_plf)
fitted(fit_plf)
residuals(fit_plf)

# iteratively reweighted least squares for 2PL model
M <- formulaNLR(model = "2PL", parameterization = "logistic")
(fit_irls <- estimNLR(
  y = y, match = match, group = group,
  formula = M$M1$formula, method = "irls"
))

coef(fit_irls)
logLik(fit_irls)
vcov(fit_irls)
fitted(fit_irls)
residuals(fit_irls)
}
\references{
Drabinova, A. & Martinkova, P. (2017). Detection of differential item
functioning with nonlinear regression: A non-IRT approach accounting for
guessing. Journal of Educational Measurement, 54(4), 498--517,
\doi{10.1111/jedm.12158}.

Hladka, A. & Martinkova, P. (2020). difNLR: Generalized logistic regression
models for DIF and DDF detection. The R Journal, 12(1), 300--323,
\doi{10.32614/RJ-2020-014}.

Hladka, A. (2021). Statistical models for detection of differential item
functioning. Dissertation thesis.
Faculty of Mathematics and Physics, Charles University.

Hladka, A., Martinkova, P., & Brabec, M. (2025). New iterative algorithms
for estimation of item functioning. Journal of Educational and Behavioral
Statistics. Online first, \doi{10.3102/10769986241312354}.
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{hladka@cs.cas.cz} \cr

Patricia Martinkova \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{martinkova@cs.cas.cz} \cr
}
\keyword{DIF}
