% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/difORD.R
\name{difORD}
\alias{difORD}
\alias{print.difORD}
\alias{plot.difORD}
\alias{coef.difORD}
\alias{logLik.difORD}
\alias{AIC.difORD}
\alias{BIC.difORD}
\title{DIF detection for ordinal data.}
\usage{
difORD(Data, group, focal.name, model = "adjacent", type = "both", match = "zscore",
anchor = NULL, purify = FALSE, nrIter = 10, p.adjust.method = "none",
parametrization = "irt", alpha = 0.05)

\method{print}{difORD}(x, ...)

\method{coef}{difORD}(object, SE = FALSE, simplify = FALSE, ...)

\method{logLik}{difORD}(object, item = "all", ...)

\method{AIC}{difORD}(object, item = "all", ...)

\method{BIC}{difORD}(object, item = "all", ...)

\method{plot}{difORD}(x, item = "all", title, plot.type, group.names, ...)
}
\arguments{
\item{Data}{data.frame or matrix: dataset which rows represent ordinaly scored examinee answers and
columns correspond to the items. In addition, \code{Data} can hold the vector of group membership.}

\item{group}{numeric or character: a dichotomous vector of the same length as \code{nrow(Data)}
or a column identifier of \code{Data}.}

\item{focal.name}{numeric or character: indicates the level of \code{group} which corresponds to
focal group.}

\item{model}{character: logistic regression model for ordinal data (either \code{"adjacent"} (default)
or \code{"cumulative"}). See \strong{Details}.}

\item{type}{character: type of DIF to be tested. Either \code{"both"} for uniform and non-uniform
DIF (i.e., difference in parameters \code{"a"} and \code{"b"}) (default), or \code{"udif"} for
uniform DIF only (i.e., difference in difficulty parameter \code{"b"}), or \code{"nudif"} for
non-uniform DIF only (i.e., difference in discrimination parameter \code{"a"}). Can be specified
as a single value (for all items) or as an item-specific vector.}

\item{match}{numeric or character: matching criterion to be used as an estimate of trait. Can be
either \code{"zscore"} (default, standardized total score), \code{"score"} (total test score),
or vector of the same length as number of observations in \code{Data}.}

\item{anchor}{numeric or character: specification of DIF free items. Either \code{NULL} (default),
or a vector of item names (column names of \code{Data}), or item identifiers (integers specifying
the column number) determining which items are currently considered as anchor (DIF free) items.
Argument is ignored if \code{match} is not \code{"zscore"} or \code{"score"}.}

\item{purify}{logical: should the item purification be applied? (default is \code{FALSE}).}

\item{nrIter}{numeric: the maximal number of iterations in the item purification (default is 10).}

\item{p.adjust.method}{character: method for multiple comparison correction. Possible values are
\code{"holm"}, \code{"hochberg"}, \code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"},
\code{"fdr"}, and \code{"none"} (default). For more details see \code{\link[stats]{p.adjust}}.}

\item{parametrization}{character: parametrization of regression coefficients. Possible options are
\code{"irt"} for difficulty-discrimination parametrization (default) and \code{"classic"} for
intercept-slope parametrization. See \strong{Details}.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{x}{an object of \code{"difORD"} class.}

\item{...}{other generic parameters for S3 methods.}

\item{object}{an object of \code{"difORD"} class.}

\item{SE}{logical: should the standard errors of estimated parameters be also returned? (default is \code{FALSE}).}

\item{simplify}{logical: should the estimated parameters be simplified to a matrix? (default is \code{FALSE}).}

\item{item}{numeric or character: either the vector of column indicator (number or column name) or \code{'all'}
(default) for all items.}

\item{title}{string: title of a plot.}

\item{plot.type}{character: which plot should be displayed for cumulative logit regression model. Either
\code{"category"} (default) for category probabilities or \code{"cumulative"} for cumulative probabilities.}

\item{group.names}{character: names of reference and focal group.}
}
\value{
The \code{difORD()} function returns an object of class \code{"difORD"}. The output
including values of the test statistics, p-values, and items marked as DIF is displayed by the
\code{print()} method.

Item characteristic curves can be displayed with \code{plot()} method.
Estimated parameters can be displayed with \code{coef()} method.

Log-likelihood, Akaike's information criterion, and Schwarz's Bayesian criterion can be
extracted with methods \code{logLik()}, \code{AIC()}, \code{BIC()} for converged item(s)
specified in \code{item} argument.

A list of class \code{"difORD"} with the following arguments:
\describe{
  \item{\code{Sval}}{the values of likelihood ratio test statistics.}
  \item{\code{ordPAR}}{the estimates of the final model.}
  \item{\code{ordSE}}{standard errors of the estimates of the final model.}
  \item{\code{parM0}}{the estimates of null model.}
  \item{\code{parM1}}{the estimates of alternative model.}
  \item{\code{llM0}}{log-likelihood of null model.}
  \item{\code{llM1}}{log-likelihood of alternative model.}
  \item{\code{AICM0}}{AIC of null model.}
  \item{\code{AICM1}}{AIC of alternative model.}
  \item{\code{BICM0}}{BIC of null model.}
  \item{\code{BICM1}}{BIC of alternative model.}
  \item{\code{DIFitems}}{either the column identifiers of the items which were detected as DIF, or
  \code{"No DIF item detected"} in case no item was detected as DIF.}
  \item{\code{model}}{model used for DIF detection.}
  \item{\code{type}}{character: type of DIF that was tested.}
  \item{\code{parametrization}}{Parameters' parametrization.}
  \item{\code{purification}}{\code{purify} value.}
  \item{\code{nrPur}}{number of iterations in item purification process. Returned only if \code{purify}
  is \code{TRUE}.}
  \item{\code{difPur}}{a binary matrix with one row per iteration of item purification and one column per item.
  \code{"1"} in i-th row and j-th column means that j-th item was identified as DIF in i-th iteration. Returned only
  if \code{purify} is \code{TRUE}.}
  \item{\code{conv.puri}}{logical indicating whether item purification process converged before the maximal number
  \code{nrIter} of iterations. Returned only if \code{purify} is \code{TRUE}.}
  \item{\code{p.adjust.method}}{character: method for multiple comparison correction which was applied.}
  \item{\code{pval}}{the p-values by likelihood ratio test.}
  \item{\code{adj.pval}}{the adjusted p-values by likelihood ratio test using \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of likelihood ratio test.}
  \item{\code{alpha}}{numeric: significance level.}
  \item{\code{Data}}{the data matrix.}
  \item{\code{group}}{the vector of group membership.}
  \item{\code{group.names}}{levels of grouping variable.}
  \item{\code{match}}{matching criterion.}
  }
}
\description{
Performs DIF detection procedure for ordinal data based either on adjacent category logit
model or on cumulative logit model and likelihood ratio test of a submodel.
}
\details{
Performs DIF detection procedure for ordinal data based either on adjacent category logit model
or on cumulative logit model.

Using adjacent category logit model, logarithm of ratio of probabilities of two adjacent
categories is
\deqn{log(P(y = k)/P(y = k-1)) = (a + aDif*g)*(x - b_k - b_kDif*g),}
where \eqn{x} is by default standardized total score (also called Z-score) and \eqn{g} is a group
membership. Parameter \eqn{a} is a discrimination of the item and parameter \eqn{b_k} is difficulty
for the \eqn{k}-th category of the item. Terms \eqn{a_Dif} and \eqn{b_kDif} then represent differences
between two groups (reference and focal) in relevant parameters.

Using cumulative logit model, probability of gaining at least \eqn{k} points is given by
2PL model, i.e.,
\deqn{P(y >= k) = exp((a + aDif*g)*(x - b_k - b_kDif*g))/(1 + exp((a + aDif*g)*(x - b_k - b_kDif*g))).}
The category probability (i.e., probability of gaining exactly \eqn{k} points) is then
\eqn{P(Y = k) = P(Y >= k) - P(Y >= k + 1)}.

Both models are estimated by iteratively reweighted least squares. For more details see \code{\link[VGAM]{vglm}}.

Argument \code{parametrization} is a character which specifies parametrization of regression parameters.
Default option is \code{"irt"} which returns IRT parametrization (difficulty-discrimination, see above).
Option \code{"classic"} returns intercept-slope parametrization with effect of group membership and
interaction with matching criterion, i.e. \eqn{b_0k + b_1*x + b_2k*g + b_3*x*g} instead of
\eqn{(a + a_Dif*g)*(x - b_k - b_kDif*g))}.

Missing values are allowed but discarded for item estimation. They must be coded as \code{NA}
for both, \code{Data} and \code{group} parameters.
}
\examples{
# loading data
data(dataMedicalgraded, package = "ShinyItemAnalysis")
Data <- dataMedicalgraded[, 1:5]
group <- dataMedicalgraded[, 101]

# Testing both DIF effects with adjacent category logit model
(x <- difORD(Data, group, focal.name = 1, model = "adjacent"))

# graphical devices
plot(x, item = 3)
\dontrun{
plot(x, item = "X2003")
plot(x, item = "X2003", group.names = c("Group 1", "Group 2"))

# estimated parameters
coef(x)
coef(x, SE = TRUE) # with SE
coef(x, SE = TRUE, simplify = TRUE) # with SE, simplified

# AIC, BIC, log-likelihood
AIC(x)
BIC(x)
logLik(x)
}
# AIC, BIC, log-likelihood for the first item
AIC(x, item = 1)
BIC(x, item = 1)
logLik(x, item = 1)
\dontrun{
# Testing both DIF effects with Benjamini-Hochberg adjustment method
difORD(Data, group, focal.name = 1, model = "adjacent", p.adjust.method = "BH")

# Testing both DIF effects with item purification
difORD(Data, group, focal.name = 1, model = "adjacent", purify = TRUE)

# Testing uniform DIF effects
difORD(Data, group, focal.name = 1, model = "adjacent", type = "udif")
# Testing non-uniform DIF effects
difORD(Data, group, focal.name = 1, model = "adjacent", type = "nudif")

# Testing both DIF effects with total score as matching criterion
difORD(Data, group, focal.name = 1, model = "adjacent", match = "score")

# Testing both DIF effects with cumulative logit model
# using IRT parametrization
(x <- difORD(Data, group, focal.name = 1, model = "cumulative", parametrization = "irt"))

# graphical devices
plot(x, item = 3, plot.type = "cumulative")
plot(x, item = 3, plot.type = "category")

# estimated parameters in IRT parametrization
coef(x, simplify = TRUE)
}
}
\references{
Agresti, A. (2010). Analysis of ordinal categorical data. Second edition. John Wiley & Sons.
}
\seealso{
\code{\link[stats]{p.adjust}} \code{\link[VGAM]{vglm}}
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
\email{hladka@cs.cas.cz} \cr

Patricia Martinkova \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{martinkova@cs.cas.cz} \cr
}
\keyword{DIF}
