% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter-data.R
\name{filter-data}
\alias{filter-data}
\alias{filter_data_ui}
\alias{filter_data_server}
\title{Shiny module to interactively filter a \code{data.frame}}
\usage{
filter_data_ui(id, show_nrow = TRUE, max_height = NULL)

filter_data_server(
  id,
  data = reactive(NULL),
  vars = reactive(NULL),
  name = reactive("data"),
  drop_ids = TRUE,
  widget_char = c("select", "picker"),
  widget_num = c("slider", "range"),
  widget_date = c("slider", "range"),
  label_na = "NA"
)
}
\arguments{
\item{id}{Module id. See \code{\link[shiny]{callModule}}.}

\item{show_nrow}{Show number of filtered rows and total.}

\item{max_height}{Maximum height for filters panel, useful
if you have many variables to filter and limited space.}

\item{data}{\code{\link[shiny]{reactive}} function returning a
\code{data.frame} to filter.}

\item{vars}{\code{\link[shiny]{reactive}} function returning a
\code{character} vector of variables for which to add a filter.
If a named \code{list}, names are used as labels.}

\item{name}{\code{\link[shiny]{reactive}} function returning a
\code{character} string representing \code{data} name, only used for code generated.}

\item{drop_ids}{Drop columns containing more than 90\% of unique values, or than 50 distinct values.}

\item{widget_char}{Widget to use for \code{character} variables: \code{\link[shinyWidgets:pickerInput]{shinyWidgets::pickerInput}}
or  \code{\link[shiny:selectInput]{shiny::selectizeInput}} (default).}

\item{widget_num}{Widget to use for \code{numeric} variables: \code{\link[shinyWidgets:numericRangeInput]{shinyWidgets::numericRangeInput}}
or  \code{\link[shiny:sliderInput]{shiny::sliderInput}} (default).}

\item{widget_date}{Widget to use for \code{date/time} variables: \code{\link[shiny:dateRangeInput]{shiny::dateRangeInput}}
or  \code{\link[shiny:sliderInput]{shiny::sliderInput}} (default).}

\item{label_na}{Label for missing value widget.}
}
\value{
\itemize{
\item UI: HTML tags that can be included in shiny's UI
\item Server: a \code{list} with three slots:
\itemize{
\item \strong{filtered}: a \code{reactive} function returning the data filtered.
\item \strong{code}: a \code{reactive} function returning the dplyr pipeline to filter data.
\item \strong{expr}: a \code{reactive} function returning an expression to filter data.
}
}
}
\description{
Module generate inputs to filter \code{data.frame} according column's type.
Code to reproduce the filter is returned as an expression with filtered data.
}
\examples{
library(shiny)
library(shinyWidgets)
library(datamods)
library(MASS)

# Add some NAs to mpg
mtcars_na <- mtcars
mtcars_na[] <- lapply(
  X = mtcars_na,
  FUN = function(x) {
    x[sample.int(n = length(x), size = sample(15:30, 1))] <- NA
    x
  }
)

datetime <- data.frame(
  date = seq(Sys.Date(), by = "day", length.out = 300),
  datetime = seq(Sys.time(), by = "hour", length.out = 300),
  num = sample.int(1e5, 300)
)

ui <- fluidPage(
  tags$h2("Filter data.frame"),

  radioButtons(
    inputId = "dataset",
    label = "Data:",
    choices = c(
      "iris",
      "mtcars",
      "mtcars_na",
      "Cars93",
      "datetime"
    ),
    inline = TRUE
  ),

  fluidRow(
    column(
      width = 3,
      filter_data_ui("filtering", max_height = "500px")
    ),
    column(
      width = 9,
      progressBar(
        id = "pbar", value = 100,
        total = 100, display_pct = TRUE
      ),
      DT::dataTableOutput(outputId = "table"),
      tags$b("Code dplyr:"),
      verbatimTextOutput(outputId = "code_dplyr"),
      tags$b("Expression:"),
      verbatimTextOutput(outputId = "code"),
      tags$b("Filtered data:"),
      verbatimTextOutput(outputId = "res_str")
    )
  )
)

server <- function(input, output, session) {

  data <- reactive({
    get(input$dataset)
  })

  vars <- reactive({
    if (identical(input$dataset, "mtcars")) {
      setNames(as.list(names(mtcars)[1:5]), c(
        "Miles/(US) gallon",
        "Number of cylinders",
        "Displacement (cu.in.)",
        "Gross horsepower",
        "Rear axle ratio"
      ))
    } else {
      NULL
    }
  })

  res_filter <- filter_data_server(
    id = "filtering",
    data = data,
    name = reactive(input$dataset),
    vars = vars,
    widget_num = "slider",
    widget_date = "slider",
    label_na = "Missing"
  )

  observeEvent(res_filter$filtered(), {
    updateProgressBar(
      session = session, id = "pbar",
      value = nrow(res_filter$filtered()), total = nrow(data())
    )
  })

  output$table <- DT::renderDT({
    res_filter$filtered()
  }, options = list(pageLength = 5))


  output$code_dplyr <- renderPrint({
    res_filter$code()
  })
  output$code <- renderPrint({
    res_filter$expr()
  })

  output$res_str <- renderPrint({
    str(res_filter$filtered())
  })

}

if (interactive())
  shinyApp(ui, server)
}
