% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{dabest}
\alias{dabest}
\title{Prepare Data for Analysis with dabestr}
\usage{
dabest(.data, x, y, idx, paired = FALSE, id.column = NULL)
}
\arguments{
\item{.data}{A data.frame or tibble.}

\item{x, y}{Columns in \code{.data}.}

\item{idx}{A vector containing factors or strings in the \code{x} columns.
These must be quoted (ie. surrounded by quotation marks). The first element
will be the control group, so all differences will be computed for every
other group and this first group.}

\item{paired}{Boolean, default FALSE. If TRUE, the two groups are treated as
paired samples. The first group is treated as pre-intervention and the
second group is considered post-intervention.}

\item{id.column}{Default NULL. A column name indicating the identity of the
datapoint if the data is paired. \emph{This must be supplied if paired is
\code{TRUE}.}}
}
\value{
A \code{dabest} object with 8 elements.

 \describe{

 \item{\code{data}}{ The dataset passed to \code{\link{dabest}}, stored here
 as a \code{\link[tibble]{tibble}}. }

 \item{\code{x} and \code{y}}{ The columns in \code{data} used to plot the x
 and y axes, respectively, as supplied to \code{\link{dabest}}. These are
 \href{https://adv-r.hadley.nz/quasiquotation.html}{quoted variables} for
 \href{https://tidyeval.tidyverse.org/}{tidy evaluation} during the
 computation of effect sizes. }

 \item{\code{idx}}{ The vector of control-test groupings. For each pair in
 \code{idx}, an effect size will be computed by downstream \code{dabestr}
 functions used to compute \link[=mean_diff]{effect sizes} (such as
 \code{mean_diff()}.  }

 \item{\code{is.paired}}{ Whether or not the experiment consists of paired
 (aka repeated) observations. }

 \item{\code{id.column}}{ If \code{is.paired} is \code{TRUE}, the column in
 \code{data} that indicates the pairing of observations. }

 \item{\code{.data.name}}{ The variable name of the dataset passed to
 \code{\link{dabest}}. }

 \item{\code{.all.groups}}{ All groups as indicated in the \code{idx}
 argument. }

 }
}
\description{
\code{dabest} prepares a
\href{https://vita.had.co.nz/papers/tidy-data.pdf}{tidy dataset} for analysis
using estimation statistics.
}
\details{
Estimation statistics is a statistical framework that focuses on effect sizes
and confidence intervals around them, rather than \emph{P} values and
associated dichotomous hypothesis testing.

\code{dabest}() collates the data in preparation for the computation of
\link[=mean_diff]{effect sizes}. Bootstrap resampling is used to compute
non-parametric assumption-free confidence intervals. Visualization of the
effect sizes and their confidence intervals using estimation plots is then
performed with a specialized \link[=plot.dabest_effsize]{plotting} function.
}
\examples{
# Performing unpaired (two independent groups) analysis.
unpaired_mean_diff <- dabest(iris, Species, Petal.Width,
                             idx = c("setosa", "versicolor"),
                             paired = FALSE)

# Display the results in a user-friendly format.
unpaired_mean_diff

# Compute the mean difference.
mean_diff(unpaired_mean_diff)

# Plotting the mean differences.
mean_diff(unpaired_mean_diff) \%>\% plot()

# Performing paired analysis.
# First, we munge the `iris` dataset so we can perform a within-subject
# comparison of sepal length vs. sepal width.

new.iris     <- iris
new.iris$ID  <- 1: length(new.iris)
setosa.only  <-
  new.iris \%>\%
  tidyr::gather(key = Metric, value = Value, -ID, -Species) \%>\%
  dplyr::filter(Species \%in\% c("setosa"))

paired_mean_diff <- dabest(setosa.only, Metric, Value,
                           idx = c("Sepal.Length", "Sepal.Width"),
                           paired = TRUE, id.col = ID) \%>\%
                    mean_diff()



# Using pipes to munge your data and then passing to `dabest`.
# First, we generate some synthetic data.
set.seed(12345)
N        <- 70
c         <- rnorm(N, mean = 50, sd = 20)
t1        <- rnorm(N, mean = 200, sd = 20)
t2        <- rnorm(N, mean = 100, sd = 70)
long.data <- tibble::tibble(Control = c, Test1 = t1, Test2 = t2)

# Munge the data using `gather`, then pass it directly to `dabest`

meandiff <- long.data \%>\%
              tidyr::gather(key = Group, value = Measurement) \%>\%
              dabest(x = Group, y = Measurement,
                     idx = c("Control", "Test1", "Test2"),
                     paired = FALSE) \%>\%
              mean_diff()



}
\seealso{
\itemize{

 \item \link[=mean_diff]{Effect size computation} from the loaded data.

 \item \link[=plot.dabest_effsize]{Generating estimation plots} after effect size computation.

 }
}
