% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setup_trial.R
\name{setup_trial}
\alias{setup_trial}
\title{Setup a generic trial specification}
\usage{
setup_trial(
  arms,
  true_ys,
  fun_y_gen = NULL,
  fun_draws = NULL,
  start_probs = NULL,
  fixed_probs = NULL,
  min_probs = rep(NA, length(arms)),
  max_probs = rep(NA, length(arms)),
  data_looks = NULL,
  max_n = NULL,
  look_after_every = NULL,
  control = NULL,
  control_prob_fixed = NULL,
  inferiority = 0.01,
  superiority = 0.99,
  equivalence_prob = NULL,
  equivalence_diff = NULL,
  equivalence_only_first = NULL,
  futility_prob = NULL,
  futility_diff = NULL,
  futility_only_first = NULL,
  highest_is_best = FALSE,
  soften_power = 1,
  fun_raw_est = mean,
  cri_width = 0.95,
  n_draws = 5000,
  robust = TRUE,
  description = NULL,
  add_info = NULL
)
}
\arguments{
\item{arms}{character vector with unique names for the trial arms.}

\item{true_ys}{numeric vector specifying true outcomes (e.g., event
probabilities, mean values, etc.) for all trial \code{arms}.}

\item{fun_y_gen}{function, generates outcomes. See \link{setup_trial} \strong{Details}
for information on how to specify this function.\cr
\strong{Note:} this function is called once during setup to validate the output
structure.}

\item{fun_draws}{function, generates posterior draws. See \link{setup_trial}
\strong{Details} for information on how to specify this function.\cr
\strong{Note:} this function is called up to three times during setup to
validate the output structure.}

\item{start_probs}{numeric vector, allocation probabilities for each arm at
the beginning of the trial. The default (\code{NULL}) is automatically
changed to equal randomisation.}

\item{fixed_probs}{numeric vector, fixed allocation probabilities for each
arm - must be either a numeric vector with \code{NA} for arms without fixed
probabilities and values between \code{0} and \code{1} for the other arms or \code{NULL}
(default), if adaptive randomisation is used for all arms or if one of the
special settings (\code{"sqrt-based"}, \code{"sqrt-based start"},
\code{"sqrt-based fixed"}, or \code{"match"}) is specified for \code{control_prob_fixed}
(described below).}

\item{min_probs}{numeric vector, lower threshold for adaptive allocation
probabilities, lower probabilities will be rounded up to these values. Must
be \code{NA} (default for all arms) if no boundary is wanted.}

\item{max_probs}{numeric vector, upper threshold for adaptive allocation
probabilities, higher probabilities will be rounded down to these values.
Must be \code{NA} (default for all arms) if no boundary is wanted.}

\item{data_looks}{vector of increasing integers, specifies when to conduct
adaptive analyses (= the total number of patients randomised at each
adaptive analysis). The last number in the vector represents the maximum
sample size. Instead of specifying \code{data_looks}, the \code{max_n} and
\code{look_after_every} arguments can be used in combination (then \code{data_looks}
must be \code{NULL}, the default).}

\item{max_n}{single integer, maximum total sample size (defaults to \code{NULL}).
Must only be specified if \code{data_looks} is \code{NULL}. Requires specification of
the \code{look_after_every} argument.}

\item{look_after_every}{single integer, specified together with \code{max_n}.
Adaptive analyses will be conducted after every \code{look_after_every}
patients randomised, and at the total sample size as specified by \code{max_n}
(\code{max_n} does not need to be a multiple of \code{look_after_every}). If
specified, \code{data_looks} must be \code{NULL} (as default).}

\item{control}{single character string, name of one of the \code{arms} or \code{NULL}
(default). If specified, this arm will serve as a common control arm, to
which all other arms will be compared and the
inferiority/superiority/equivalence thresholds (see below) will be for
those comparisons. See \link{setup_trial} \strong{Details} below for information on
behaviour with respect to these comparisons.}

\item{control_prob_fixed}{if a common \code{control} arm is specified, this must
be set to either \code{NULL} (the default), in which case the control arm
allocation probability will not be fixed if control arms change (the
allocation probability to the first control arm may still be fixed using
\code{fixed_probs}) Otherwise a vector of probabilities of either length \code{1} or
\verb{number of arms - 1} can be provided, or one of the special arguments
\code{"sqrt-based"}, \code{"sqrt-based start"}, \code{"sqrt-based fixed"} or \code{"match"}.
See \link{setup_trial} \strong{Details} below for details in behaviour.}

\item{inferiority}{single numeric (\verb{> 0} and \verb{<1}, default is \code{0.01})
specifying the inferiority threshold. An arm will be considered inferior
and dropped if the probability that it is best (when comparing all arms) or
better than the control arm (when a common \code{control} is used) drops below
this threshold.}

\item{superiority}{single numeric (\verb{>0} and \verb{<1}, default is \code{0.99})
specifying the superiority threshold. If the probability that an arm is
best (when comparing all arms) or better than the control arm (when a
common \code{control} is used) exceeds this number, said arm will be declared
the winner and the trial will be stopped (if no common \code{control} is used or
if the last comparator is dropped in a design with a common control) \emph{or}
become the new control and the trial will continue (if a common control is
specified).}

\item{equivalence_prob}{single numeric (\verb{> 0} and \verb{< 1}) or \code{NULL} (default,
corresponding to no equivalence assessment). If a numeric value is
specified, arms will be stopped for equivalence if the probability of
either \emph{(a)} equivalence compared to a common \code{control} or \emph{(b)}
equivalence between all arms remaining (designs without a common control)
exceeds this threshold. Requires specification of \code{equivalence_diff},
\code{equivalence_only_first}, and a common \code{control} arm.}

\item{equivalence_diff}{single numeric value (\verb{> 0}) or \code{NULL} (default,
corresponding to no equivalence assessment). If a numeric value is
specified, estimated differences below this threshold will be considered
equivalent when assessing equivalence. For designs with a common \code{control}
arm, the differences between each non-control arm and the \code{control} arm is
used, and for trials without a common \code{control} arm, the difference between
the highest and lowest estimated outcome rates are used and the trial is
only stopped for equivalence if all remaining arms are thus equivalent.}

\item{equivalence_only_first}{single logical in trial specifications where
\code{equivalence_prob} and \code{equivalence_diff} are specified, otherwise \code{NULL}
(default). Must be \code{NULL} for designs without a common control arm. If a
common \code{control} arm is used, this specifies whether equivalence will only
be assessed for the first control (if \code{TRUE}) or also for subsequent
control arms (if \code{FALSE}) if one arm is superior to the first control and
becomes the new control.}

\item{futility_prob}{single numeric (\verb{> 0} and \verb{< 1}) or \code{NULL} (default,
corresponds to no futility assessment). If a numeric value is specified,
\code{arms} will be stopped for futility when the probability for futility
compared to the common \code{control} exceeds this threshold. Requires a common
\code{control} arm, specification of \code{futility_diff} and \code{futility_only_first}.}

\item{futility_diff}{single numeric value (\verb{> 0}) or \code{NULL} (default,
corresponding to no futility assessment). If a numeric value is specified,
estimated differences below this threshold in the \emph{beneficial} direction
(as specified in \code{highest_is_best}) will be considered futile when
assessing futility in designs with a common \code{control} arm. If only 1 arm
remains after dropping arms for futility, the trial will be stopped without
declaring the last arm superior.}

\item{futility_only_first}{single logical in trial specifications designs
where \code{futility_prob} and \code{futility_diff} are specified, otherwise \verb{NULL (default). Must be }NULL\verb{for designs without a common}control\verb{arm. Specifies whether futility will only be assessed against the first}control\verb{(if}TRUE\verb{) or also for subsequent control arms (if }FALSE`) if
one arm is superior to the first control and becomes the new control.}

\item{highest_is_best}{single logical, specifies whether larger estimates of
the outcome are favourable or not; defaults to \code{FALSE}, corresponding to,
e.g., an undesirable binary outcomes (e.g., mortality) or a continuous
outcome where lower numbers are preferred (e.g., hospital length of stay).}

\item{soften_power}{either a single numeric value or a numeric vector of
exactly the same length as the maximum number of looks/adaptive analyses.
Values must be between \code{0} and \code{1} (default); if \verb{< 1}, then re-allocated
non-fixed allocation probabilities are all raised to this power to make
allocation probabilities less extreme, in turn used to redistribute
remaining probability while respecting limits when defined by \code{min_probs}
and/or \code{max_probs}. If \code{1}, then no \emph{softening} is applied.}

\item{fun_raw_est}{function that takes a numeric vector and returns a
single numeric value, used to calculate a raw summary estimate of the
outcomes in each \code{arm}. Defaults to \link{mean}, which is always used in the
\link{setup_trial_binom} and \link{setup_trial_norm} functions.\cr
\strong{Note:} the function is called one time per arm during setup to validate
the output structure.}

\item{cri_width}{single numeric \verb{>= 0} and \verb{< 1}, the width of the
percentile-based credible intervals used when summarising individual trial
results. Defaults to \code{0.95}, corresponding to 95\% credible intervals.}

\item{n_draws}{single integer, the number of draws from the posterior
distributions (for each arm) used when running the trial. Defaults to
\code{5000}; can be reduced for a speed gain (at the potential loss of stability
of results if too low) or increased for increased precision (takes longer).
Values \verb{< 100} are not allowed and values \verb{< 1000} are not recommended
and warned against.}

\item{robust}{single logical, if \code{TRUE} (default) the medians and median
absolute deviations (scaled to be comparable to the standard deviation for
normal distributions; MAD_SD) are used to summarise the posterior
distributions; if \code{FALSE}, the means and standard deviations (SDs) are used
instead (slightly faster, but may be less appropriate for posteriors skewed
on the natural scale).}

\item{description}{optional single character string describing the trial
design, will only be used in print functions if not \code{NULL} (the default).}

\item{add_info}{optional single string containing additional information
regarding the trial design or specifications, will only be used in print
functions if not \code{NULL} (the default).}
}
\value{
A \code{trial_spec} object used to run simulations by \link{run_trial} or
\link{run_trials}. The output is essentially a list containing the input values
(some combined in a \code{data.frame} called \code{trial_arms}), but its class
signals that these inputs have been validated and inappropriate
combinations and settings have been ruled out. Also contains \code{best_arm}
holding the arm(s) with the best value(s) in \code{true_ys}. Use \code{str()} to
peruse the actual content of the returned object
}
\description{
Specifies the design of an adaptive trial with any type of outcome and
validates all inputs. Use \link{run_trial} or \link{run_trials} to conduct
single/multiple simulations of the specified trial, respectively.\cr
See \link{setup_trial_binom} and \link{setup_trial_norm} for simplified setup of
trial designs common outcome types. For additional trial specification
examples, see the the \strong{Basic examples} vignette
(\code{vignette("Basic-examples", package = "adaptr")}) and the
\strong{Advanced example} vignette
(\code{vignette("Advanced-example", package = "adaptr")}).
}
\details{
\strong{How to specify the \code{fun_y_gen} function}

The function must take the following inputs:
\itemize{
\item \code{allocs}: character vector, the trial \code{arms} that new patients allocated
since the last adaptive analysis are randomised to.
}

The function must return a single numeric vector, corresponding to the
outcomes for all patients allocated since the last adaptive analysis, in the
same order as \code{allocs}.\cr
See the \strong{Examples} vignette (\code{vignette("Examples", "adaptr")}) for an
example with further details.

\strong{How to specify the \code{fun_draws} function}

The function must take the following inputs:
\itemize{
\item \code{arms}: character vector, the unique trial arms, in the same order as
above, but only the \strong{currently active} arms are specified when the function is
called.
\item \code{allocs}: a vector of allocations for all patients, corresponding to the
trial arms, including patients allocated to \strong{currently inactive} \code{arms} when
called,
\item \code{ys}: a vector of outcomes for all patients in the same order as \code{allocs},
including outcomes for patients allocated to \strong{currently inactive} \code{arms}
when called.
\item \code{control}: single character, the current \verb{control arm}, will be \code{NULL} for
designs without a common control arm, but required regardless as the argument
is supplied by \link{run_trial}.
\item \code{n_draws}: single integer, the number of posterior draws for each arm.
}

The function must return a \code{matrix} (with numeric values) with \code{arms} columns
and \code{n_draws} rows. The \code{matrix} must have columns
\strong{only for currently active arms} (when called). Each row should contain a
single posterior draw for each arm on the original outcome
scale: if they are estimated as, e.g., the \emph{log(odds)}, these estimates must
be transformed to probabilities and similarly for other measures.\cr
Important: the \code{matrix} cannot contain \code{NA}s, even if no patients have been
randomised to an arm yet. See the provided example for one way to alleviate
this.
\cr
See the \strong{Examples} vignette (\code{vignette("Examples", "adaptr")}) for an
example with further details.

\emph{Notes}
\itemize{
\item Different estimation methods and prior distributions may be used;
complex functions will lead to slower simulations compared to simpler
methods for obtaining posterior draws, including those specified using the
\link{setup_trial_binom} and \link{setup_trial_norm} functions.
\item Technically, using log relative effect measures — e.g. log(odds ratio) or
log(risk ratios) - or differences compared to a reference arm (e.g., mean
differences or absolute risk differences) instead of absolute values in each
arm will work to some extent (\strong{be cautious!}):
\item Stopping for superiority/inferiority/max sample sizes will work.
\item Stopping for equivalence/futility may be used with relative effect
measures on the log scale.
\item Several summary statistics from \link{run_trial} (\code{sum_ys} and posterior
estimates) may be nonsensical if relative effect measures are used
(depending on calculation method).
\item In the same vein, \link{extract_results} (\code{sum_ys}, \code{sq_err}, and
\code{sq_err_te}), and \link{summary} (\code{sum_ys_mean/sd/median/q25/q75}, \code{rmse},
\code{rmse_te} and \code{idp}) may be equally nonsensical when calculated on the
relative scale.
}

\strong{More information on arguments}
\itemize{
\item \code{control}: if one or more treatment arms are superior to the control arm
(i.e., passes the superiority threshold as defined above), this arm will
become the new control (if multiple arms are superior, the one with the
highest probability of being the overall best will become the new control),
the previous control will be dropped for inferiority, and all remaining arms
will be immediately compared to the new control in the same adaptive analysis
and dropped if inferior (or possibly equivalent/futile, see below) compared
to this new control arm. Only applies in trials with a common \code{control}.
\item \code{control_prob_fixed}: If the length is 1, then this allocation probability
will be used for the \code{control} group (including if a new arm becomes the
control and the original control is dropped). If multiple values are specified
the first value will be used when all arms are active, the second when one
arm has been dropped, and so forth. If 1 or more values are specified,
previously set \code{fixed_probs}, \code{min_probs} or \code{max_probs} for new control arms
will be ignored. If all allocation probabilities do not sum to 1 (e.g, due to
multiple limits) they will be re-scaled to do so.\cr
Can also be set to one of the special arguments \code{"sqrt-based"},
\code{"sqrt-based start"}, \code{"sqrt-based fixed"} or \code{"match"} (written exactly as
one of those, case sensitive). This requires \code{start_probs} to be \code{NULL} and
relevant \code{fixed_probs} to be \code{NULL} (or \code{NA} for the control arm).\cr
If one of the \code{"sqrt-based"/"sqrt-based start"/"sqrt-based fixed"} options
are used, the function will set \emph{square-root-transformation-based} starting
allocation probabilities. These are defined as:\cr
\verb{square root of number of non-control arms to 1-ratio for other arms}\cr
scaled to sum to 1, which will generally increase power for comparisons
against the common control, as discussed in, e.g., \emph{Park et al, 2020}
\doi{10.1016/j.jclinepi.2020.04.025}.\cr
If \code{"sqrt-based"}, square-root-transformation-based allocation probabilities
will also be used for new controls when arms are dropped. If
\code{"sqrt-based start"}, the control arm will be fixed to this allocation
probability at all times (also after arm dropping, with re-scaling as
necessary, as specified above). If \code{"sqrt-based fixed"} is chosen,
square-root-transformation-based allocation probabilities will be used and
all allocation probabilities will be fixed throughout the trial (with
re-scaling when arms are dropped).\cr
If \code{"match"} is specified, the control group allocation will always be
\emph{matched} to be similar to the highest non-control arm allocation ratio.
}

\strong{Superiority and inferiority}

In trial designs without a common control arm, superiority and inferiority
are assessed by comparing all \emph{\strong{currently active}} groups. This means that
if a "final" analysis of a trial without a common control and \verb{> 2 arms} is
conducted including all arms (as will often be done in practice) \emph{after} an
adaptive trial have stopped, the final probabilities of the best arm being
superior may differ slightly.\cr
For example, in a trial with three arms and no common \code{control} arm, one arm
may be dropped early for inferiority defined as \verb{< 1\%} probability of being
the overall best \code{arm}. The trial may then continue with the two remaining
arms, and stopped when one is declared superior to the other defined as
\verb{> 99\%} probability of being the overall best \code{arm}. If a final analysis is
then conducted including all arms, the final probability of the best arm
being overall superior will generally be slightly lower as the probability
of the first dropped arm being the best will generally be \verb{> 0\%}, even if
very low and below the inferiority threshold.\cr
This is not relevant trial designs \emph{with} a common \code{control}, as pairwise
assessments of superiority/inferiority compared to the common \code{control} will
not be influenced similarly by previously dropped arms (and previously
dropped arms may be included in the analyses, even if posterior distributions
are not returned for those).
Similarly, in actual clinical trials, final probabilities may change slightly
as the most recently randomised patients will generally not have outcome data
available at the final adaptive analysis where the trial is stopped.

\strong{Equivalence}

Equivalence is assessed \emph{\strong{after}} both inferiority and
superiority have been assessed (and in case of superiority, it will be
assessed against the new \code{control} arm in designs with a common \code{control}, if
specified - see above).

\strong{Futility}

Futility is assessed \emph{\strong{after}} inferiority, superiority, \emph{\strong{and}}
equivalence have been assessed (and in case of superiority, it will be
assessed against the new control arm in designs with a common control, if
specified - see above). Arms will thus be dropped for equivalence before
futility.
}
\examples{
# Setup a custom trial specification with right-skewed, log-normally
# distributed continuous outcomes (higher values are worse)

# Define the function that will generate the outcomes in each arm
# Notice: contents should match arms/true_ys in the setup_trial call below
get_ys_lognorm <- function(allocs) {
  y <- numeric(length(allocs))
  # arms (names and order) and values (except for exponentiation) should match
  # those used in setup_trial (below)
  means <- c("Control" = 2.2, "Experimental A" = 2.1, "Experimental B" = 2.3)
  for (arm in names(means)) {
    ii <- which(allocs == arm)
    y[ii] <- rlnorm(length(ii), means[arm], 1.5)
  }
  y
}

# Define the function that will generate posterior draws
# In this example, the function uses no priors (corresponding to improper
# flat priors) and calculates results on the log-scale, before exponentiating
# back to the natural scale, which is required for assessments of
# equivalence, futility and general interpretation
get_draws_lognorm <- function(arms, allocs, ys, control, n_draws) {
  draws <- list()
  logys <- log(ys)
  for (arm in arms){
    ii <- which(allocs == arm)
    n <- length(ii)
    if (n > 1) {
      # Necessary to avoid errors if too few patients randomised to this arm
      draws[[arm]] <- exp(rnorm(n_draws, mean = mean(logys[ii]), sd = sd(logys[ii])/sqrt(n - 1)))
    } else {
      # Too few patients randomised to this arm - extreme uncertainty
      draws[[arm]] <- exp(rnorm(n_draws, mean = mean(logys), sd = 1000 * (max(logys) - min(logys))))
    }
  }
  do.call(cbind, draws)
}

# The actual trial specification is then defined
lognorm_trial <- setup_trial(
  # arms should match those above
  arms = c("Control", "Experimental A", "Experimental B"),
  # true_ys should match those above
  true_ys = exp(c(2.2, 2.1, 2.3)),
  fun_y_gen = get_ys_lognorm, # as specified above
  fun_draws = get_draws_lognorm, # as specified above
  max_n = 5000,
  look_after_every = 200,
  control = "Control",
  # Qquare-root-based, fixed control group allocation ratio
  # and response-adaptive randomisation for other arms
  control_prob_fixed = "sqrt-based",
  # Equivalence assessment
  equivalence_prob = 0.9,
  equivalence_diff = 0.5,
  equivalence_only_first = TRUE,
  highest_is_best = FALSE,
  # Summarise raw results by taking the mean on the
  # log scale and back-transforming
  fun_raw_est = function(x) exp(mean(log(x))) ,
  # Summarise posteriors using medians with MAD-SDs,
  # as distributions will not be normal on the actual scale
  robust = TRUE,
  # Description/additional info used when printing
  description = "continuous, log-normally distributed outcome",
  add_info = "SD on the log scale for all arms: 1.5"
)

# Print trial specification with 3 digits for all probabilities
print(lognorm_trial, prob_digits = 3)

}
