% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statistical_simulation.r
\name{simbase_covar}
\alias{simbase_covar}
\title{Calculate reference data for simulating values based on a covariance matrix approach}
\usage{
simbase_covar(
  data,
  variables = NULL,
  transforms = list(),
  label = simbase_labeler,
  ...
)
}
\arguments{
\item{data}{The dataset for the calculation of the reference data for
simulation; for grouped datasets (see \code{\link{group_by}}),
the reference data is
calculated for each group separately (see also \code{\link{simbase_list}}).}

\item{variables}{Character vector containing the names in \code{data}
which should be included in the simulation. If missing, all numeric
variables in \code{data} are used.}

\item{transforms}{A named list of objects of class \code{trans}
or class \code{transform}
(see function \code{trans_new} in package \code{scales});
the name of each list entry
\bold{must} correspond to a variable name in \code{variables}.}

\item{label}{Either a string describing the data and the simulation approach,
or a labelling function which returns a label string and takes as input
the data, a string giving the class
of the simbase object (here \code{"simbase_covar"}) and the
transforms list.}

\item{...}{Arguments to be passed on to \code{\link{simbase_list}}
(\emph{if} it is called).}
}
\value{
An \code{S3} object of class \code{simbase_list} if \code{data} is grouped,
and an object of class \code{simbase_covar} otherwise.
}
\description{
Given the covariance matrix and the means of a set of variables, we can
simulate not only the distribution of the variables, but also their
correlations. The present function calculates the basic values required for
the simulation and returns them packed into an object of class
\code{simbase_covar}.
}
\details{
If some of the variables are non-normally distributed, a transform may
improve the prediction. The transforms are passed to the function as a named
list, where the name of a list entry must correspond to the name of the
variable in the data which is to be transformed.

Predefined transforms can be found in the package \code{scales}, where they are
used for axis transformations as a preparation for plotting. The package
\code{scales} also contains a function \code{trans_new} which can be used
to define new transforms.

In the context of destructively measured sawn timber properties, the type of
destructive test applied is of interest. If the dataset \code{data} contains a
variable \code{loadtype} which consistently throughout the dataset has either the
value "t" (i.e. all sawn timber has been tested in tension) or the
value "be" (i.e. all sawn timber has been tested in bending, edgewise),
then the returned object also has a field \code{loadtype} with that value.

One can also calculate a simbase under the assumption that the correlations
are different for different subgroups of the data. This is done by grouping
the dataset \code{data} prior to passing it to the function,
using \code{\link{group_by}}. In this case, several objects of
class \code{simbase_covar} are created and joined together in a \code{\link{tibble}} --
see also \code{\link{simbase_list}}.
}
\examples{
# obtain a dataset for demonstration
dataset <- simulate_dataset();

# calculate a simbase without transforms
simbase_covar(dataset, c('f', 'E', 'rho', 'E_dyn'));

# calculate a simbase with log-transformed f
simbase_covar(dataset, c('f', 'E', 'rho', 'E_dyn'), list(f = scales::log_trans()));

# if we group the dataset, we get a simbase_list object
simbase_covar(dplyr::group_by(dataset, country), c('f', 'E', 'rho', 'E_dyn'));

}
