#' Unimodal Non-Parametric Cluster (UNPaC) Significance Test
#'
#' The UnPAC test assesses the significance of clusters by comparing the cluster index (CI) from the data to the CI from a ortho-unimodal reference data generated using a Gaussian copula.
#' This method is similar to them method described in Helgeson and Bair (2016) except a Gaussian copula approach is used to account for feature correlation.
#'  The CI is defined to be the sum of the
#' within-cluster sum of squares about the cluster means divided by the total sum of squares. Smaller values of the
#' CI indicate a stronger clustering.
#'
#' @param x a dataset with n observations (rows) and p features (columns)
#' @param cluster labels generated by clustering method
#' @param cluster.fun function used to cluster data. Function should return list containing a component "cluster."
#' Examples include \code{\link[stats]{kmeans}} and  \code{\link[cluster]{pam}}.
#' @param var_selection should dimension be reduced using feature filtering procedure? See description below. (default=FALSE)
#' @param gamma threshold for feature filtering procedure. See description below. Not used if var_selection=FALSE (default=0.10)
#' @param p.adjust p-value adjustment method for additional feature filtering. See \code{\link[stats]{p.adjust}}
#' for options. (default="fdr"). Not used if p.adjust="none."
#' @param nsim a numeric value specifying the number of unimodal reference distributions used for testing
#' (default=100)
#' @param k integer value specifying the number of clusters to test (default=2)
#' @param rho  a regularization parameter used in implementation of the graphical lasso. See documentation for lambda in
#' \code{\link[huge]{huge}}.
#' Not used if \code{cov="est"} or \code{cov="banded"}
#' @param cov method used for approximating the covariance structure.  options include: "glasso"
#' (See \code{\link[huge]{huge}}), "banded"  (See \code{\link[PDSCE]{band.chol.cv}}) and
#'        "est" (default = "glasso")
#' @param scale should data be scaled such that each feature has variance equal to one prior to clustering
#' (default=FALSE)
#' @param center should data be centered such that each feature has mean equal to zero prior to clustering
#' (default=TRUE)
#'
#' @return
#' The function returns a list with the following components:
#' \itemize{
#' \item{\code{selected_features}}: {A vector of integers indicating the features retained by the feature filtering process.}
#' \item{\code{sim_CI}}: {vector containing the cluster indices for each generated unimodal reference distribution}
#' \item{\code{pvalue_emp}}: {the empirical p-value:  the proportion of times the cluster index from the reference
#' data is smaller the cluster index from the observed data}
#' \item{\code{pvalue_norm}}: {the normalized p-value: the simulated p-value based on comparison to a standard normal distribution }
#' }
#'
#' @references
#' \itemize{
#'     \item Helgeson E and Bair E (2016). ``Non-Parametric Cluster Significance Testing with Reference to a Unimodal Null Distribution."
#'     arXiv preprint arXiv:1610.01424.
#'     \item Rothman, A. J., Levina, E., and Zhu, J. (2010). ``A new approach to Cholesky-based covariance regularization in
#'  high dimensions." Biometrika 97(3): 539-550.
#' }
#'
#'
#' @details
#' There are three options for the covariance matrix used in generating the Gaussian
#' copula: sample covariance estimation, \code{cov="est"}, which should be used if n>p; the graphical lasso,
#' \code{cov="glasso"}, which should be used if n<p; and  k-banded covariance, \code{cov="banded"}, which can be used if n<p and it can be assumed that
#' features farther away in the ordering have weaker covariance. The graphical lasso is implemented using the \code{\link[huge]{huge}} function.
#' When \code{cov="banded"} is selected the k-banded covariance Cholesky factor of Rothman, Levina, and Zhu (2010) is used to estimate the covariance matrix.
#' Cross-validation is used for selecting the banding parameter. See documentation in \code{\link[PDSCE]{band.chol.cv}}.
#'
#' In high dimensional (n<p) settings a dimension reduction step can be implemented which selects features
#' based on an F-test for difference in means across clusters. Features having a p-value less than a threshold
#' \code{gamma} are retained. For additional feature filtering a p-value adjustment procedure (such as p.adjust="fdr")
#' can be used. If no features are retained the resulting p-value for the cluster significance test is given as 1.
#'
#'
#'
#' @examples
#' # K-means example
#' test1 <- matrix(rnorm(100*50), nrow=100, ncol=50)
#' test1[1:30,1:50] <- rnorm(30*50, 2)
#' test.data<-scale(test1,scale=FALSE,center=TRUE)
#' cluster<-kmeans(test.data,2)$cluster
#' UNPaCResults <- UNPaC_Copula(test.data,cluster,kmeans, nsim=100,cov="est")
#'
#' # Hierarchical clustering example
#'  \donttest{
#' test <- matrix(nrow=1200, ncol=75)
#' theta <- rep(NA, 1200)
#' theta[1:500] <- runif(500, 0, pi)
#' theta[501:1200] <- runif(700, pi, 2*pi)
#' test[1:500,seq(from=2,to=50,by=2)] <- -2+5*sin(theta[1:500])
#' test[501:1200,seq(from=2,to=50,by=2)] <- 5*sin(theta[501:1200])
#' test[1:500,seq(from=1,to=49,by=2)] <- 5+5*cos(theta[1:500])
#' test[501:1200,seq(from=1,to=49,by=2)] <- 5*cos(theta[501:1200])
#' test[,1:50] <- test[,1:50] + rnorm(50*1200, 0, 0.2)
#' test[,51:75] <- rnorm(25*1200, 0, 1)
#' test.data<-scale(test,center=TRUE,scale=FALSE)
#' # Defining clustering function
#' hclustFunction<-function(x,k){
#'  D<-dist(x)
#'  xn.hc <- hclust(D, method="single")
#'  list(cluster=cutree(xn.hc, k))}
#'
#' cluster=hclustFunction(test.data,2)$cluster
#' UNPaCResults <- UNPaC_Copula(test.data,cluster,hclustFunction, nsim=100,cov="est")
#' }
#' @export
#' @name UNPaC_Copula
#' @author Erika S. Helgeson, David Vock, Eric Bair

UNPaC_Copula <-
  function(x,cluster,cluster.fun, nsim=100,var_selection=FALSE,gamma=0.1,p.adjust="fdr", k=2, rho=0.02,cov="glasso",center=TRUE,scale=FALSE) {
  out <- rep(NA, nsim)
  x<-scale(x,scale=scale,center=center)
  selected_features=NA
  if (var_selection==TRUE){
    selected_features=unique(unlist(var.selection(cluster.fun,cluster,k=k,x=x,p.adjust,gamma)))
    if (length(selected_features)==0){print("No selected features")
      return(list(selected_features=NA,sim_CI=NA,pvalue_emp=1,pvalue_norm=1))
      }else{
     x<-matrix(x[,selected_features],ncol=length(selected_features))
    cluster <- cluster.fun(x, k)$cluster
  }}

  test.km.ci <- CI(x,cluster)

  out=UNPaC_null_sig(x, k,cluster.fun,nsim=nsim, rho=rho, cov=cov,center=center,scale=scale)
  pvalue_emp=sum(test.km.ci>out) / length(out)
  pvalue_norm=pnorm((test.km.ci-mean(out))/sd(out))
  return(list(selected_features=selected_features,sim_CI=out,pvalue_emp=pvalue_emp,pvalue_norm=pvalue_norm))
}

