% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shrink.R
\name{shrink}
\alias{shrink}
\title{Shrinking characteristics of precision matrix estimators}
\usage{
shrink(X = NULL, Y = NULL, S = NULL, A = diag(ncol(S)),
  B = diag(ncol(S)), C = matrix(0, ncol = ncol(B), nrow = ncol(A)),
  nlam = 10, lam.max = NULL, lam.min.ratio = 0.001, lam = NULL,
  alpha = 1, path = FALSE, rho = 2, mu = 10, tau.rho = 2,
  iter.rho = 10, crit = c("ADMM", "loglik"), tol.abs = 1e-04,
  tol.rel = 1e-04, maxit = 10000, adjmaxit = NULL, K = 5,
  crit.cv = c("MSE", "loglik", "penloglik", "AIC", "BIC"), start = c("warm",
  "cold"), cores = 1, trace = c("progress", "print", "none"))
}
\arguments{
\item{X}{option to provide a nxp data matrix. Each row corresponds to a single observation and each column contains n observations of a single feature/variable.}

\item{Y}{option to provide nxr response matrix. Each row corresponds to a single response and each column contains n response of a single feature/response.}

\item{S}{option to provide a pxp sample covariance matrix (denominator n). If argument is \code{NULL} and \code{X} is provided instead then \code{S} will be computed automatically.}

\item{A}{option to provide user-specified matrix for penalty term. This matrix must have p columns. Defaults to identity matrix.}

\item{B}{option to provide user-specified matrix for penalty term. This matrix must have p rows. Defaults to identity matrix.}

\item{C}{option to provide user-specified matrix for penalty term. This matrix must have nrow(A) rows and ncol(B) columns. Defaults to zero matrix.}

\item{nlam}{number of \code{lam} tuning parameters for penalty term generated from \code{lam.min.ratio} and \code{lam.max} (automatically generated). Defaults to 10.}

\item{lam.max}{option to specify the maximum \code{lam} tuning parameter. Defaults to NULL.}

\item{lam.min.ratio}{smallest \code{lam} value provided as a fraction of \code{lam.max}. The function will automatically generate \code{nlam} tuning parameters from \code{lam.min.ratio*lam.max} to \code{lam.max} in log10 scale. If \code{lam.max = NULL}, \code{lam.max} is calculated to be the smallest \code{lam} such that all off-diagonal entries in \code{Omega} are equal to zero. Defaults to 1e-3.}

\item{lam}{option to provide positive tuning parameters for penalty term. This will cause \code{nlam} and \code{lam.min.ratio} to be disregarded. If a vector of parameters is provided, they should be in increasing order. Defaults to NULL.}

\item{alpha}{elastic net mixing parameter contained in [0, 1]. \code{0 = ridge, 1 = lasso}. Alpha must be a single value (cross validation across alpha not supported).}

\item{path}{option to return the regularization path. This option should be used with extreme care if the dimension is large. If set to TRUE, cores must be set to 1 and errors and optimal tuning parameters will based on the full sample. Defaults to FALSE.}

\item{rho}{initial step size for ADMM algorithm.}

\item{mu}{factor for primal and residual norms in the ADMM algorithm. This will be used to adjust the step size \code{rho} after each iteration.}

\item{tau.rho}{factor in which to increase/decrease step size \code{rho}}

\item{iter.rho}{step size \code{rho} will be updated every \code{iter.rho} steps}

\item{crit}{criterion for convergence (\code{ADMM} or \code{loglik}). If \code{crit = loglik} then iterations will stop when the relative change in log-likelihood is less than \code{tol.abs}. Default is \code{ADMM} and follows the procedure outlined in Boyd, et al.}

\item{tol.abs}{absolute convergence tolerance. Defaults to 1e-4.}

\item{tol.rel}{relative convergence tolerance. Defaults to 1e-4.}

\item{maxit}{maximum number of iterations. Defaults to 1e4.}

\item{adjmaxit}{adjusted maximum number of iterations. During cross validation this option allows the user to adjust the maximum number of iterations after the first \code{lam} tuning parameter has converged (for each \code{alpha}). This option is intended to be paired with \code{warm} starts and allows for 'one-step' estimators. Defaults to NULL.}

\item{K}{specify the number of folds for cross validation.}

\item{crit.cv}{cross validation criterion (\code{MSE}, \code{loglik}, \code{penloglik}, \code{AIC}, or \code{BIC}). Defaults to \code{MSE}.}

\item{start}{specify \code{warm} or \code{cold} start for cross validation. Default is \code{warm}.}

\item{cores}{option to run CV in parallel. Defaults to \code{cores = 1}.}

\item{trace}{option to display progress of CV. Choose one of \code{progress} to print a progress bar, \code{print} to print completed tuning parameters, or \code{none}.}
}
\value{
returns class object \code{ADMMsigma} which includes:
\item{Call}{function call.}
\item{Iterations}{number of iterations.}
\item{Tuning}{optimal tuning parameter.}
\item{Lambdas}{grid of lambda values for CV.}
\item{maxit}{maximum number of iterations.}
\item{Omega}{estimated penalized precision matrix.}
\item{Sigma}{estimated covariance matrix from the penalized precision matrix (inverse of Omega).}
\item{Path}{array containing the solution path. Solutions will be ordered in ascending alpha values for each lambda.}
\item{Z}{final sparse update of estimated penalized precision matrix.}
\item{Y}{final dual update.}
\item{rho}{final step size.}
\item{Loglik}{penalized log-likelihood for Omega}
\item{MIN.error}{minimum average cross validation error (cv.crit) for optimal parameters.}
\item{AVG.error}{average cross validation error (cv.crit) across all folds.}
\item{CV.error}{cross validation errors (cv.crit).}
}
\description{
Shrinking characteristics of precision matrix estimators. Penalized precision matrix estimation using the ADMM algorithm.
Consider the case where \eqn{X_{1}, ..., X_{n}} are iid \eqn{N_{p}(\mu,
\Sigma)} and we are tasked with estimating the precision matrix,
denoted \eqn{\Omega \equiv \Sigma^{-1}}. This function solves the
following optimization problem:
\describe{
\item{Objective:}{
\eqn{\hat{\Omega}_{\lambda} = \arg\min_{\Omega \in S_{+}^{p}}
\left\{ Tr\left(S\Omega\right) - \log \det\left(\Omega \right) +
\lambda\left\| A \Omega B - C \right\|_{1} \right\}}}
}
where \eqn{\lambda > 0} and we define
\eqn{\left\|A \right\|_{1} = \sum_{i, j} \left| A_{ij} \right|}.
}
\details{
For details on the implementation of 'shrink', see the vignette
\url{https://mgallow.github.io/SCPME/}.
}
\examples{
# generate some data
data = data_gen(n = 100, p = 5, r = 1)

# lasso penalized omega (glasso)
shrink(X = data$X, Y = data$Y)

# lasso penalized beta (print estimated omega)
lam.max = max(abs(t(data$X) \%*\% data$Y))
(shrink = shrink(X = data$X, Y = data$Y, B = cov(data$X, data$Y), lam.max = lam.max))

# print estimated beta
shrink$Z
}
\references{
\itemize{
\item Boyd, Stephen, Neal Parikh, Eric Chu, Borja Peleato, Jonathan Eckstein, and others. 2011. 'Distributed Optimization and Statistical Learning via the Alternating Direction Method of Multipliers.' \emph{Foundations and Trends in Machine Learning} 3 (1). Now Publishers, Inc.: 1-122. \url{https://web.stanford.edu/~boyd/papers/pdf/admm_distr_stats.pdf}
\item Hu, Yue, Chi, Eric C, amd Allen, Genevera I. 2016. 'ADMM Algorithmic Regularization Paths for Sparse Statistical Machine Learning.' \emph{Splitting Methods in Communication, Imaging, Science, and Engineering}. Springer: 433-459.
\item Molstad, Aaron J., and Adam J. Rothman. (2017). 'Shrinking Characteristics of Precision Matrix Estimators. \emph{Biometrika.}. \url{https://doi.org/10.1093/biomet/asy023}
\item Rothman, Adam. 2017. 'STAT 8931 notes on an algorithm to compute the Lasso-penalized Gaussian likelihood precision matrix estimator.'
}
}
\seealso{
\code{\link{plot.shrink}}
}
\author{
Matt Galloway \email{gall0441@umn.edu}
}
