% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RNAmf_two_level.R
\name{RNAmf_two_level}
\alias{RNAmf_two_level}
\title{Fitting the Recursive non-additive model with two fidelity levels.}
\usage{
RNAmf_two_level(X1, y1, X2, y2, kernel = "sqex", constant = TRUE, ...)
}
\arguments{
\item{X1}{vector or matrix of input locations for the low fidelity level.}

\item{y1}{vector of response values for the low fidelity level.}

\item{X2}{vector or matrix of input locations for the high fidelity level.}

\item{y2}{vector of response values for the high fidelity level.}

\item{kernel}{character specifying kernel type to be used, to be chosen between \code{"sqex"}(squared exponential), \code{"matern1.5"}, or \code{"matern2.5"}. Default is \code{"sqex"}.}

\item{constant}{logical indicating for constant mean of GP (\code{constant=TRUE}) or zero mean (\code{constant=FALSE}). Default is \code{TRUE}.}

\item{...}{for compatibility with \code{optim}.}
}
\value{
\itemize{
  \item \code{fit1}: list of fitted model for \eqn{(X_1, y_1)}.
  \item \code{fit2}: list of fitted model for \eqn{((X_2, f_1(X_2)), y_2)}.
  \item \code{constant}: copy of \code{constant}.
  \item \code{kernel}: copy of \code{kernel}.
  \item \code{level}: a level of the fidelity. It returns 2.
  \item \code{time}: a scalar of the time for the computation.
}
}
\description{
The function fits RNA models with designs of two fidelity levels.
The estimation method is based on MLE.
Possible kernel choices are squared exponential, Matern kernel with smoothness parameter 1.5 and 2.5.
The function returns fitted model at level 1 and 2, whether constant mean or not, and kernel choice.
}
\details{
Consider the model
\eqn{\begin{cases}
& f_1(\bm{x}) = W_1(\bm{x}),\\
& f_2(\bm{x}) = W_2(\bm{x}, f_1(\bm{x})),
\end{cases}}
where \eqn{f_l} is the simulation code at fidelity level \eqn{l}, and
\eqn{W_l(\bm{x}) \sim GP(\alpha_l, \tau_l^2 K_l(\bm{x}, \bm{x}'))} is GP model.
Hyperparameters \eqn{(\alpha_l, \tau_l^2, \bm{\theta_l})} are estimated by
maximizing the log-likelihood via an optimization algorithm "L-BFGS-B".
For \code{constant=FALSE}, \eqn{\alpha_l=0}.

Covariance kernel is defined as:
\eqn{K_l(\bm{x}, \bm{x}')=\prod^d_{j=1}\phi(x_j,x'_j;\theta_{lj})} with
\eqn{\phi(x, x';\theta) = \exp \left( -\frac{ \left( x - x' \right)^2}{\theta}  \right)}
for squared exponential kernel; \code{kernel="sqex"},
\eqn{\phi(x,x';\theta) =\left( 1+\frac{\sqrt{3}|x- x'|}{\theta} \right) \exp \left( -\frac{\sqrt{3}|x- x'|}{\theta} \right)}
for Matern kernel with the smoothness parameter of 1.5; \code{kernel="matern1.5"} and
\eqn{\phi(x, x';\theta) =  \left( 1+\frac{\sqrt{5}|x-x'|}{\theta} +\frac{5(x-x')^2}{3\theta^2} \right) \exp \left( -\frac{\sqrt{5}|x-x'|}{\theta} \right)}
for Matern kernel with the smoothness parameter of 2.5; \code{kernel="matern2.5"}.

For details, see Heo and Sung (2023+, <arXiv:2309.11772>).
}
\examples{
\donttest{
### two levels example ###
library(lhs)

### Perdikaris function ###
f1 <- function(x) {
  sin(8 * pi * x)
}

f2 <- function(x) {
  (x - sqrt(2)) * (sin(8 * pi * x))^2
}

### training data ###
n1 <- 13
n2 <- 8

### fix seed to reproduce the result ###
set.seed(1)

### generate initial nested design ###
X <- NestedX(c(n1, n2), 1)
X1 <- X[[1]]
X2 <- X[[2]]

### n1 and n2 might be changed from NestedX ###
### assign n1 and n2 again ###
n1 <- nrow(X1)
n2 <- nrow(X2)

y1 <- f1(X1)
y2 <- f2(X2)

### fit an RNAmf ###
fit.RNAmf <- RNAmf_two_level(X1, y1, X2, y2, kernel = "sqex")
}

}
\seealso{
\code{\link{predict.RNAmf}} for prediction.
}
