###########################################################################/**
# @RdocDocumentation toNNN
# @alias toBMP
# @alias toEMF
# @alias toEPS
# @alias toPDF
# %% @alias toCairoPNG
# @alias toPNG
# @alias toSVG
# @alias toTIFF
# @alias toWMF
# @alias toFavicon
# @alias asDataURI
#
# % Interactive/screen devices
# @alias toDefault
# @alias toQuartz
# @alias toWindows
# @alias toX11
# @alias toCairoWin
# @alias toCairoX11
# %% @alias toJavaGD
#
# @title "Methods for creating image files of a specific format"
#
# \description{
#  @get "title".
# }
#
# \usage{
#   toBMP(name, ...)
#   toPDF(name, ...)
#   toPNG(name, ...)
#   toSVG(name, ...)
#   toTIFF(name, ...)
#   toEMF(name, ..., ext="emf")
#   toWMF(name, ..., ext="wmf")
#
#   toFavicon(..., name="favicon", ext="png",
#             field=getDevOption("favicon", "field", default="htmlscript"))
#
#   % Interactive/screen devices
#   toDefault(name, ...)
#   toQuartz(name, ...)
#   toX11(name, ...)
#   toWindows(name, ...)
#
#   % Interactive/screen devices from other packages
#   toCairoWin(name, ...)
#   toCairoX11(name, ...)
#   %% toJavaGD(name, ...)
# }
#
# \arguments{
#   \item{name}{A @character string specifying the name of the image file.}
#   \item{...}{Additional arguments passed to @see "devEval", e.g.
#      \code{tags} and \code{aspectRatio}.}
#   \item{ext, field}{Passed to @see "devEval".}
# }
#
# \value{
#   Returns by default the @see "DevEvalProduct".
#   For \code{toFavicon()} the default return value is a @character string.
# }
#
# \details{
#   Both \code{toEMF()} and \code{toWMF()} use the exact same graphics
#   device and settings.  They only differ by filename exension.
# }
#
# @author
#
# \seealso{
#   These functions are wrappers for @see "devEval".
#   See @see "devOptions" to change the default dimensions for
#   a specific device type.
# }
#
# @keyword device
# @keyword utilities
#*/###########################################################################
toBMP <- function(name, ...) {
  devEval(type="bmp", name=name, ...);
}

toEMF <- function(name, ..., ext="emf") {
  devEval(type="win.metafile", name=name, ..., ext=ext);
}

toWMF <- function(name, ..., ext="wmf") {
  devEval(type="win.metafile", name=name, ..., ext=ext);
}

toEPS <- function(name, ...) {
  devEval(type="eps", name=name, ...);
}

toPDF <- function(name, ...) {
  devEval(type="pdf", name=name, ...);
}

toPNG <- function(name, ...) {
  devEval(type="png", name=name, ...);
}

toSVG <- function(name, ...) {
  devEval(type="svg", name=name, ...);
}

toTIFF <- function(name, ...) {
  devEval(type="tiff", name=name, ...);
}

toFavicon <- function(..., name="favicon", ext="png", field=getDevOption("favicon", "field", default="htmlscript")) {
  # Output as "<script>...</script>"?
  if (identical(field, "htmlscript")) {
    uri <- devEval(type="favicon", name=name, ext=ext, field="dataURI", ...);
    script <- c(
      "<script>",
      " var link = document.createElement('link');",
      " link.rel = 'icon';",
      sprintf(' link.href = "%s"', uri),
      " document.getElementsByTagName('head')[0].appendChild(link);",
      "</script>",
      ""
    )
    paste(script, collapse="\n")
  } else {
    devEval(type="favicon", name=name, ext=ext, field=field, ...);
  }
} # toFavicon()

asDataURI <- function(..., mime=NULL) {
  img <- DevEvalFileProduct(...);
  if (is.null(mime)) {
    mime <- getMimeType(img);
  }
  getDataURI(img, mime=mime);
}


toDefault <- function(name, ...) {
  devEval(type=getOption("device"), name=name, ...);
}

toQuartz <- function(name, ...) {
  devEval(type="quartz", name=name, ...);
}

toWindows <- function(name, ...) {
  devEval(type="windows", name=name, ...);
}

toX11 <- function(name, ...) {
  devEval(type="X11", name=name, ...);
}

toCairoWin <- function(name, ...) {
  devEval(type="CairoWin", name=name, ...);
}

toCairoX11 <- function(name, ...) {
  devEval(type="CairoX11", name=name, ...);
}

## toJavaGD <- function(name, ...) {
##   devEval(type="JavaGD", name=name, ...);
## }





##############################################################################
# HISTORY:
# 2014-09-15
# o Added support for toFavicon(..., field="htmlscript").  The device
#   options are such that the default is to use "htmlscript".
# o CLEANUP: toNnn():s no longer sets 'field' explitly.
# 2014-08-29
# o Changed default argument 'field' for toNNN() from "fullname" to
#   "pathname" as in getOption("devEval/args/field", "pathname").
#   This should work for most compilers include LaTeX, although in
#   later it is preferred to use "fullname".
# 2014-04-28
# o Waiting with toCairoPNG() and toJavaGD().
# o Added toDefault(), toX11(), toWindows(), and toQuartz().
#   Also added toCairoWin() and toCairoX11() for the 'Cairo' package.
# 2014-01-11
# o Added asDataURI().
# 2014-01-10
# o Added toFavicon().
# 2013-07-24
# o Added toEMF()/toWMF().
# 2013-02-23
# o Changed default argument 'field' for toNNN() from "fullname"
#   to getOption("devEval/args/field", "fullname").
# 2012-05-01
# o Renamed Rd to toNNN.
# 2012-04-27
# o Renamed Rd to toPNG().
# 2012-04-07
# o Added toBMP() and toTIFF().
# 2011-11-05
# o Added toSVG().
# o Moved from private R.rsp.addons package to R.utils.
# o Dropped default 'width' from toPNG(), toPDF() and toEPS().  No longer
#   required from R.utils v1.9.1.
# 2011-10-31
# o CLEANUP: toPNG() etc now utilizes new 'field' argument of devEval().
# 2011-09-12
# o Added toEPS().
# 2011-06-15
# o Added argument 'field' to toPNG() and toPDF().
# 2011-05-23
# o Added toPNG() and toPDF().
##############################################################################
