#' Calculate sample size needed to achieve a prespecified power
#'
#' @importFrom stats qnorm
#' @importFrom stats pnorm
#' @param tau the estimated treatment effect $E[Y(1) - Y(0)]$
#' @param sig.level the significance level
#' @param power the power to achieve; if left NULL and sample.size is not NULL, will return the corresponding power given sample.size
#' @param r the proportion of treated units
#' @param phi the overlap coefficients
#' @param E1,E0,S1,S0,R1,R0 the summary quantities
#' @param sample.size sample size to calculate power; ignored when power is not NULL
#' @param test whether one-sided or two-sided test is considered
#' @param estimand the estimand (ATE, ATT, ATC or ATO), or a customized tilting function
#' @returns an object with the calculated sample size
#' @examples
#' PSpower(1, 0.05, 0.956, 0.5, 0.99, -1.74, -2.74, 19.86, 20.12, 0.14, 0.14)
#' @export
PSpower <- function(tau, sig.level, power = NULL, r, phi, E1, E0, S1, S0, R1, R0,
                    sample.size = NULL,
                    test = 'two-sided', estimand = 'ATE') {
  params <- solve_parameters(r, phi, E1, E0, S1, S0, R1, R0)
  if (typeof(estimand) == 'character') {
    if(!estimand %in% c('ATE', 'ATT', 'ATC', 'ATO'))
      stop('estimand should be one of ATE, ATT, ATC and ATO, or the tilting function.')
    if (estimand == 'ATE') {V <- params$V_ATE}
    else if (estimand == 'ATT') {V <- params$V_ATT}
    else if (estimand == 'ATC') {V <- params$V_ATC}
    else if (estimand == 'ATO') {V <- params$V_ATO}
  }
  else if (typeof(estimand) == 'closure') {
    V <- params$calc_V(estimand)
  }
  else {
    stop('estimand should be one of ATE, ATT, ATC and ATO, or the tilting function.')
  }

  if (!is.null(power)) {
    if (test == 'two-sided') {
      coef <- (qnorm(1 - sig.level / 2) + qnorm(power))^2
    }
    else if (test == 'one-sided') {
      coef <- (qnorm(1 - sig.level) + qnorm(power))^2
    }
    else {
      stop('test should be either one-sided or two-sided.')
    }
    sample_size <- coef * V / tau^2
    power <- power
    ss_calculation <- TRUE
  }
  else {
    if (test == 'two-sided') {
      power <- pnorm(sqrt(sample.size / V) * tau - qnorm(1 - sig.level / 2))
    }
    else if (test == 'one-sided') {
      power <- pnorm(sqrt(sample.size / V) * tau - qnorm(1 - sig.level))
    }
    else {
      stop('test should be either one-sided or two-sided.')
    }
    sample_size <- sample.size
    power <- power
    ss_calculation <- FALSE
  }

  return (structure(
    list(
      sample_size = sample_size,
      tau = tau, sig.level = sig.level, power = power, test = test, estimand = estimand,
      summaries = c(r = r, phi = phi, E1 = E1, E0 = E0, S1 = S1, S0 = S0, R1 = R1, R0 = R0),
      params = params,
      ss_calculation = ss_calculation
    ),
    class = 'PSpower'
  ))
}

#' Prints PSpower object
#' @param x PSpower object
#' @param ... ignored
#' @returns no return value; called for side effect to output a string
#' @export
print.PSpower <- function(x, ...) {
  if (x$ss_calculation) {
    cat('Estimated sample size to achieve', round(x$power, 3), 'power:', round(x$sample_size), '\n')
  }
  else {
    cat('Calculated power at sample size', round(x$sample_size), 'is', round(x$power, 3), '\n')
  }
}

#' Plots PSpower object
#' @param x PSpower object
#' @param power a range of powers to plot the power curve
#' @param ... ignored
#' @importFrom ggplot2 ggplot
#' @importFrom ggplot2 aes
#' @importFrom ggplot2 geom_point
#' @importFrom ggplot2 geom_line
#' @examples
#' obj <- PSpower(1, 0.05, 0.956, 0.5, 0.99, -1.74, -2.74, 19.86, 20.12, 0.14, 0.14)
#' plot(obj)
#' @returns an object (class ggplot) containing a figure
#' @export
plot.PSpower <- function(x, power = seq(0.6, 0.99, length.out = 100), ...) {
  size <- sapply(power,
         function(p)
           PSpower(x$tau, x$sig.level, p, x$summaries['r'], x$summaries['phi'],
                   x$summaries['E1'], x$summaries['E0'], x$summaries['S1'], x$summaries['S0'],
                   x$summaries['R1'], x$summaries['R0'], sample.size = NULL, x$test, x$estimand)$sample_size
  )
  ggplot2::ggplot(data.frame(power = power, size = size), ggplot2::aes(size, power)) +
    ggplot2::geom_line() +
    ggplot2::geom_point(data = data.frame(power = x$power, size = x$sample_size))
}
