#' Translate SEA results for Mass Spectrometry searching tools
#'
#' @description This function translates singular enrichment analysis results and extracts the required information
#' for mass spectometry searching tools. The subset of protein modifications is from <https://raw.githubusercontent.com/HUPO-PSI/psi-mod-CV/master/PSI-MOD.obo>.
#'
#' @param x A dataframe of single enrichment analysis results generated by \code{\link{runEnrichment}} function.
#' @param sig.level The significance level to filter pathways (applies on adjusted p-value). Default value is 0.05.
#' @param number.rep Only consider PTM terms that occurred more than a specific number of times in UniProt. This number is set
#' by number.rep parameter. The default value is NULL.
#' @return A database of subset of protein modifications:
#' - id: a unique identification for each subset of protein modifications, PSI-MOD.
#' - name: the name of modification.
#' - def: definition of PSI-MOD definition
#'
#' @export
#'
#' @import tidyverse
#'
#' @examples
#' enrich1 <- runEnrichment(protein = exmplData1$pl1, os.name = 'Homo sapiens (Human)')
#' MS      <- sea2mass(x = enrich1, sig.level = 0.05)
sea2mass = function(x, sig.level = 0.05, number.rep = NULL){

  x <- x %>% filter(`corrected pvalue` <= sig.level)

  if( !is.null(number.rep) ){
    x <- x %>% filter(`FreqinUniprot` >= number.rep)
  }

  x <- x %>% arrange( desc(`FreqinList`) )

  pathway <- as.character(x$PTM)

  # look up each pathway in https://www.uniprot.org/docs/ptmlist.txt
  temp <- ptmlist %>% filter( ID %in% pathway )


  # print name field from https://raw.githubusercontent.com/HUPO-PSI/psi-mod-CV/master/PSI-MOD.obo
  DR      <- temp$DR
  res     <- mod_ont %>% filter(id %in% DR)

  # Remove part of the definition column
  res$def <- str_replace(string = res$def, pattern = 'A protein modification that effectively ', replacement = '')

  return(res)
}




