#######################################################################################################
#   MARSSkf function
#   Kalman filter and smoother
#   ** All eqn refs are to 2nd ed of Shumway & Stoffer (2006): Time Series Analysis and Its Applications
#######################################################################################################
MARSSkf = function(y, parList, missing.matrix = NULL, miss.value= NULL, init.state="x00", debugkf=FALSE) {
    if(is.null(missing.matrix) && is.null(miss.value)) stop("Stopped in MARSSkf() because either missing.matrix or miss.value must be specified.\n")
    if(!is.null(miss.value) && !is.na(miss.value) && !is.numeric(miss.value)) stop("Stopped in MARSSkf() because miss.value must be numeric (or NA).")
    if(!(init.state %in% c("x10","x00"))) stop("Stopped in MARSSkf() because init.state must be either x10 or x00. See user guide.\n")
    condition.limit=1E10
    condition.limit.Ft=1E5 #because the Ft is used to compute LL and LL drop limit is about 2E-8
    #Note diff in param names from S&S;B=Phi, Z=A, A not in S&S
    U=parList$U; Q=parList$Q; R=parList$R;
    A=parList$A; x0=parList$x0; V0=parList$V0; B=parList$B;
    Z=parList$Z; #this is the design matrix, called A in S&S  
	  if(length(R)==1) diag.R=unname(R) else diag.R = takediag(unname(R))
    if(length(Q)==1) diag.Q=unname(Q) else diag.Q=takediag(unname(Q));
    t.B = matrix(B,dim(B)[2],dim(B)[1],byrow=TRUE)   #faster transpose
    n=dim(y)[1]; TT=dim(y)[2]; m=dim(Q)[1]
	  M = missing.matrix
	  msg=NULL

    #Construct needed identity matrices
    I.m = diag(1,m); I.n = diag(1,n)
	  #Set up permutation matrices when there are 0s on diagonal of Q
    n.Qn0=sum(diag.Q!=0)
    n.Rn0=sum(diag.R!=0)
    if(n.Qn0==m){ 
      OmgQ=OmgQ1=t.OmgQ1=I.m
      B.inv.0 = matrix(0,m,m)
    }else{
      OmgQ1=I.m[diag.Q!=0, , drop=FALSE]
      t.OmgQ1 = t(OmgQ1)
      OmgQ0=I.m[diag.Q==0, , drop=FALSE]
      t.OmgQ0 = t(OmgQ0)
      OmgQ = t.OmgQ1 %*% OmgQ1    #if Q is all 0, this is matrix(0,m,m)
      B.inv.0 = chol2inv(chol(OmgQ0%*%B%*%t.OmgQ0))  #this part of B must be diagonal
      B.inv.0 = t.OmgQ0%*%B.inv.0%*%OmgQ0    #expand back up with 0s where Q!=0
    } #case when n.Qn0 == 0 dealt with inside the Kalman smoother
    if(n.Rn0==n){
      OmgR1=I.n
    }else{
      OmgR1=I.n[diag.R!=0, , drop=FALSE]
    }
	  #Check that if any R are 0 then model is solveable
	  OmgRVtt = I.m
	  diag.OmgRVtt = rep(1,m)
    if(any(diag.R==0)){
       Z.R0=Z[diag.R==0,,drop=FALSE]
       Z.R0.n0=Z.R0[,colSums(Z.R0)!=0,drop=FALSE] # The Z cols where there is a value
       if(dim(Z.R0.n0)[1]==dim(Z.R0.n0)[2]){ #then it must be invertable
        Ck=try(solve(Z.R0.n0))
        if(class(Ck)=="try-error"){
          return(list(ok=FALSE,error="Some R diagonal elements are 0, but Z is such that model is indeterminate in this case."))
       }else OmgRVtt = diag(ifelse(colSums(Z.R0)!=0,0,1),m) #mxm; sets the p elem of Vtt to 0 because these are determined by data
     }else if(dim(Z.R0.n0)[1]>dim(Z.R0.n0)[2]){
        return(list(ok=FALSE,error="Some R diagonal elements are 0, and Z is such that model is over-determined."))
     }
     diag.OmgRVtt = takediag(OmgRVtt)
	  }
	  
    #if the user didn't pass in the M matrix
    if(is.null(M)){
      if(is.na(miss.value)){ YM=matrix(as.numeric(!is.na(y)),n,TT)
      }else  YM=matrix(as.numeric(!(y==miss.value)),n,TT)
    }else{
      if(n==1){YM=matrix(M,1,TT)
        }else{
          YM = matrix(1,n,TT)
          for(t in 1:TT) YM[,t]=diag(M[,,t]) }
      }
        
    #Make sure the missing vals in y are zeroed out if there are any
    y[YM==0]=0
             
    #initialize matrices
    # for notation purposes, 't' represents current point in time, 'T' represents the length of the series
    Vtt = Vtt1 = VtT = Vtt1T =J = array(0,dim=c(m,m,TT))     
    # Vtt is analogous to S&S Ptt, var[xt,xt|y(1:t)];  Vtt1 is analogous to S&S Ptt1, cov[xt,xt1|y(1:t)]
    # VtT and Vtt1T are for the backwards smoother; VtT is var[xt,xt|y(1:T)]  and Vtt1T is cov[xt,xt1|y(1:T)]
    # J see eqn 6.49
    xtt = xtt1 = xtT = matrix(0,m,TT)       
    # xtt is E[x(t) | Y(t)]; xtt1 is E[x(t) | Y(t-1)]; xtT is E[x | y(1:T)]
    vt = matrix(0,n,TT)     # these are innovations, parentheses in 6.21; vt equivalent to epsilon, eqn 6.62 
    Ft = array(0,dim=c(n,n,TT))      # used for likelihood, Ft equivalent to sigma matrix eqn 6.62
    Kt = array(0, dim=c(m,n,TT))     # 3D matrix of Kalman gain, EW added 11/14/08
        
    ##############################################
    #FORWARD PASS (K filter) gets you E[x(t) given y(1:t)]
    ##############################################
    # In the following, vt and Ft are needed for the likelihood calculation
		# the missing values will contribute 0.0 for the LL calc
    # R_mod is needed for the corrected likelihood calculation when there are missing values
		# See section 12.3 in Time Series: Theory and Methods (1991) Peter J. Brockwell, Richard A. Davis
		# put 1's on the diagonal where there are missing values and zero out the rows and columns

    for (t in 1:TT) {
    #missing value modifications per S&S2006 eq 6.78
    if(any(YM[,t]==0)){
      Mt = I.n; Mt[YM[,t]==0,]=0  #much faster than makediag(YM)
      I.2 = I.n-Mt  
      Zt = Mt%*%Z
      At = Mt%*%A
      Omg1=I.n[YM[,t]==1,,drop=FALSE]
      t.Omg1 = I.n[,YM[,t]==1,drop=FALSE] 
      Rt = Mt%*%R%*%Mt + I.2%*%R%*%I.2
    }else { Zt=Z; Rt=R; At=A; Omg1=I.n; t.Omg1=I.n; Mt=I.n }
    if(length(Zt)==1) t.Zt = Zt else t.Zt = matrix(Zt,dim(Zt)[2],dim(Zt)[1],byrow=TRUE) #faster transpose
    
    #t=1 treatment depends on how you define the initial condition.  Either as x at t=1 or x at t=0
    if(t==1) {
        if(init.state=="x00") {
          xtt1[,1] = B%*%x0 + U   #Shumway and Stoffer treatment of initial states # eqn 6.19   (pi is defined as t=0)
          Vtt1[,,1] = B%*%V0%*%t.B + Q          # eqn 6.20
        }
        if(init.state=="x10") {    #Ghahramani treatment of initial states uses x10 and has no x00 (pi is defined as t=1)
         xtt1[,1] = x0         
         Vtt1[,,1] = V0
        }
    }else {   #t!=1
       xtt1[,t] = B%*%xtt[,t-1,drop=FALSE] + U  #xtt1 denotes x_t^(t-1), eqn 6.19
       Vtt1[,,t] = B%*%Vtt[,,t-1]%*%t.B + Q                  # eqn 6.20
    }
    if(m!=1) Vtt1[,,t] = (Vtt1[,,t]+matrix(Vtt1[,,t],m,m,byrow=TRUE))/2   #in general Vtt1 is not symmetric but here it is since Vtt and Q are

    #Set up the inverse needed in Kt (part corresponding to no missing values)
    if(any(YM[,t]==1)){ siginv1 = Omg1%*%(Zt%*%Vtt1[,,t]%*%t.Zt + Rt)%*%t.Omg1
    }else siginv1=I.n #placeholder siginv will be all zero in this case 
    # bracketed piece of eqn 6.23 modified per 6.78; because R diag might be 0, bracket in Omg1

    ## Catch errors before entering chol2inv
    diag.Vtt1 = unname(Vtt1[,,t]); diag.Vtt1=diag.Vtt1[1 + 0:(m - 1)*(m + 1)]   #much faster way to get the diagonal
    if(any(diag.Vtt1==0) ) {  #0s on diag of Vtt1 will break the K smoother (below) if t>1
      #deal with 0s that are ok if there are corresponding 0s on Q diagonal
      Q0s=identical(which(diag.Q==0),which(diag.Vtt1==0))
      if(!Q0s && (init.state=="x00" || (init.state=="x10" && t>1)) ){
        return(list(ok=FALSE, errors=paste("Stopped in MARSSkf: soln became unstable when zeros appeared on the diagonal of Vtt1 at t>1.\n") ) )
      }else if(any(takediag(siginv1)==0) && (init.state=="x00" || (init.state=="x10" && t>1)))
        return(list(ok=FALSE, errors=paste("Stopped in MARSSkf: soln became unstable when zeros appeared on the diagonal of siginv1[,,1].\n") ) )
    }

    if(t==1 && init.state=="x10" && identical(unname(V0),matrix(0,m,m))) {
      Kt[,,t] = V0%*%t.Zt
    }else{  #compute Kt using update equation
        siginv2=try(chol(siginv1), silent = TRUE)      
        #Catch errors before entering chol2inv
        if(class(siginv2)=="try-error") {
          Ck1 = try(kappa(siginv1))
          Ck1 = ifelse(class(Ck1)=="try-error","Inf",round(Ck1))
          Ck4 = try(kappa((diag(1,n)[diag.R!=0,])%*%R%*%t(diag(1,n)[diag.R!=0,]))) 
          Ck4 = ifelse(class(Ck4)=="try-error","Inf",round(Ck4))
          msg1=paste("Condition num. of siginv1[t=",t,"] = ",Ck1," ",sep="")
          msg2=paste("Condition num. of R = ",Ck4," ",sep="")
          return(list(ok=FALSE, errors=paste("Stopped in MARSSkf: chol(Z%*%Vtt1[,,",t,"]%*%t(Z)+R) error. ",msg1," ", msg2,"\n",sep="") ) )       
        }
        ####### End of Error-checking for this section
    
        if(n==1){ siginv = 1/siginv1 
        }else{
          siginv=chol2inv(siginv2) 
          siginv = (matrix(siginv,dim(siginv)[2],dim(siginv)[1],byrow=TRUE)+siginv)/2
        }
        if(any(YM[,t]==1)){ siginv = t.Omg1%*%siginv%*%Omg1 #expand back out with zeros in the places for missing values  
        }else siginv = matrix(0,n,n) #don't crash if all missing values
        Kt[,,t] =  Vtt1[,,t]%*%t.Zt%*%siginv
    }
    if(m==1 || n==1) Kt.tmp = matrix(Kt[,,t], m, n) else Kt.tmp = Kt[,,t] # stop R from changing matrix dim; drop=FALSE won't work here

    vt[,t] = y[,t,drop=FALSE] - (Zt%*%xtt1[,t,drop=FALSE]+At) #need to hold on to this for loglike calc
    # eqn 6.21
    xtt[,t]=xtt1[,t,drop=FALSE] + Kt.tmp%*%vt[,t,drop=FALSE]     
    Vtt[,,t] = Vtt1[,,t]-Kt.tmp%*%Zt%*%Vtt1[,,t]  # eqn 6.22, detail after 6.28, modified Z per 6.78
    if(m!=1) Vtt[,,t] = (Vtt[,,t]+matrix(Vtt[,,t],m,m,byrow=TRUE))/2 #to ensure its symetric
    OmgRVtt.t = OmgRVtt
    if(any(diag.OmgRVtt==0)) diag(OmgRVtt.t) = diag.OmgRVtt + t(!(Z==0))%*%(diag.R==0 & YM[,t]==0)
    Vtt[,,t] = OmgRVtt.t%*%Vtt[,,t]%*%OmgRVtt.t  #zero out rows cols as needed when R diag = 0
    
    # Variables needed for the likelihood calculation; see comments above
    R_mod = (I.n-Mt) + Mt%*%R%*%Mt #not in S&S; see MARSS documention per LL calc when missing values
    Ft[,,t] = Zt%*%Vtt1[,,t]%*%t.Zt+R_mod #need to hold on to this for loglike calc
    if(n!=1) Ft[,,t] = (Ft[,,t]+matrix(Ft[,,t],n,n,byrow=TRUE))/2 #to ensure its symetric
       
    ####### Error-checking
    if(debugkf) {
          Ck1 = kappa(siginv1)
          if(!all(diag.Q==0)) Ck2 = kappa(OmgQ1%*%Vtt1[,,t]%*%t(OmgQ1))  else Ck2=1
          if(!all(diag.R==0)) Ck3 = kappa(OmgR1%*%Ft[,,t]%*%t(OmgR1)) else Ck3=1
          if(Ck1>condition.limit && !all(Kt[,,t]==0) ) 
          msg=rbind(msg,paste("MARSSkf: solution is becoming unstable.  Condition num. of siginv1[t=",t,"] = ",round(Ck1),"\n",sep=""))
          if(Ck2>condition.limit && t>1) 
          msg=rbind(msg,paste("MARSSkf: solution is becoming unstable.  Condition num. of Vtt1[t=",t,"] = ",round(Ck2),"\n",sep=""))
          if( Ck3>condition.limit.Ft ){
             if(!all(diag.R==0)) Ck4 = kappa( OmgR1%*%R%*%t(OmgR1) )  else Ck4 = 1
             msg=rbind(msg,paste("MARSSkf: logLik computation is becoming unstable.  Condition num. of Sigma[t=",t,"] = ",round(Ck3)," and of R = ",round(Ck4),".\n",sep=""))
             }
          }
          #Abandon if solution is so unstable that Vtt diagonal became negative
        diag.Vtt = unname(Vtt[,,t]); diag.Vtt=diag.Vtt[1 + 0:(m - 1)*(m + 1)]   #much faster way to get the diagonal
        if(any(diag.Vtt<0) || any(diag.Vtt1<0) )
          return(list(ok=FALSE, 
          errors=paste("Stopped in MARSSkf: soln became unstable and negative values appeared on the diagonal of Vtt or Vtt1.\n") ) )
    ####### Error-checking

    } #End of the Kalman filter recursion (for i to 1:TT)

    ######################################################
    #BACKWARD PASS (Kalman smoother) gets you E[x(t)|y(1:T)] from E[x(t)|y(1:t)]
    ######################################################
    xtT[,TT] = xtt[,TT,drop=FALSE]
    VtT[,,TT] = Vtt[,,TT]
    #indexing is 0 to T for the backwards smoother recursions
    s = seq(TT,2)
    for(i in 1:(TT-1)) {
      t=s[i]
      Zt = Z; Zt[YM[,t]==0,]=0   #MUCH faster than defining Mt using diag(YM)
      #deal with any 0s on diagonal of Q
      if(n.Qn0>0){
          if(n.Qn0==1){ Vinv.sub = 1/(OmgQ1%*%Vtt1[,,t]%*%t.OmgQ1) 
          }else{ Vinv.sub = chol2inv(chol(OmgQ1%*%Vtt1[,,t]%*%t.OmgQ1))}  #OmgQ1 is dealing with 0s in Q
          Vinv = t.OmgQ1%*%Vinv.sub%*%OmgQ1  #expand back with 0s for Q=0 elements
          if(m!=1) Vinv = (Vinv + matrix(Vinv,m,m,byrow=TRUE))/2  #to enforce symmetry after chol2inv call
          J[,,t-1] = B.inv.0 + Vtt[,,t-1]%*%t.B%*%Vinv  # eqn 6.49 and 1s on diag when Q=0
        }else J[,,t-1] = B.inv.0   #Q is all zeros
      xtT[,t-1] = xtt[,t-1,drop=FALSE] + J[,,t-1]%*%(xtT[,t,drop=FALSE]-xtt1[,t,drop=FALSE])     # eqn 6.47
      if(length(J[,,t-1])==1) t.J = J[,,t-1] else t.J = matrix(J[,,t-1],m,m,byrow=TRUE) #faster transpose
      VtT[,,t-1] = Vtt[,,t-1] + J[,,t-1]%*%(VtT[,,t]-Vtt1[,,t])%*%t.J  # eqn 6.48
      #VtT[,,t-1] = (VtT[,,t-1]+matrix(VtT[,,t-1],m,m,byrow=TRUE))/2     #should not be necessary here
      }

    if(init.state=="x00") { #Shumway and Stoffer treatment of initial conditions
     if(n.Qn0>0){
          if(n.Qn0==1){ Vinv.sub = 1/(OmgQ1%*%Vtt1[,,1]%*%t.OmgQ1) 
          }else{ Vinv.sub = chol2inv(chol(OmgQ1%*%Vtt1[,,1]%*%t.OmgQ1)) }  #OmgQ1 is dealing with 0s in Q
          Vinv = t.OmgQ1%*%Vinv.sub%*%OmgQ1
          if(m!=1) Vinv = (Vinv + t(Vinv))/2 #to enforce symmetry after chol2inv call
          J0 = B.inv.0 + V0%*%t.B%*%Vinv  # eqn 6.49 ; when V0=0, this can be set to B.inv or 0 (gets same ans)
      }else J0=B.inv.0      #Q all zeros
      x0T = x0 + J0%*%(xtT[,1,drop=FALSE]-xtt1[,1,drop=FALSE]);          # eqn 6.47
      V0T = V0 + J0%*%(VtT[,,1]-Vtt1[,,1])*t(J0)   # eqn 6.48
      V0T = (V0T+t(V0T))/2;
    }
    if(init.state=="x10") { #Ghahramani treatment of initial states
      J0 = J[,,1]
      x0T = xtT[,1,drop=FALSE]
      V0T = VtT[,,1]
    }
    #LAG 1 Covariance smoother
    #run another backward recursion to get E[x(t)x(t-1)|y(T)]
    Zt = Z; Zt[YM[,TT]==0,]=0     #much faster than Mt%*%Z
    KT = matrix(Kt[,,TT], m, n); #funny array call to prevent R from restructuring dims
    Vtt1T[,,TT] = (I.m - KT%*%Zt)%*%B%*%Vtt[,,TT-1] #eqn. 6.55 this is Var(x(T)x(T-1)|y(T)); not symmetric
    s = seq(TT,3)
    for (i in 1:(TT-2)) {
       t = s[i]
       if(length(J[,,t-2])==1) t.J = J[,,t-2] else t.J = matrix(J[,,t-2],m,m,byrow=TRUE) #faster transpose
       Vtt1T[,,t-1] = Vtt[,,t-1]%*%t.J + J[,,t-1]%*%(Vtt1T[,,t]-B%*%Vtt[,,t-1])%*%t.J   #eqn 6.56
    }
    if(init.state=="x00") Vtt1T[,,1] = Vtt[,,1]%*%t(J0) + J[,,1]%*%(Vtt1T[,,2]-B%*%Vtt[,,1])%*%t(J0)
    if(init.state=="x10") Vtt1T[,,1] = NA

    ###########################################################
    #Calculate log likelihood, see eqn 6.62
    #Innovations form of the likelihood
    rtn.list = list(xtT = xtT, VtT = VtT, Vtt1T = Vtt1T, x0T = x0T, V0T = V0T, Vtt = Vtt,
            Vtt1 = Vtt1, J=J, J0=J0, Kt=Kt, xtt1 = xtt1, xtt=xtt, Innov=vt, Sigma=Ft)
    loglike = -sum(YM)/2*log(2*pi)    #sum(M) is the number of data points
    for (t in 1:TT) {
      if( t>1 && any(takediag(Ft[,,t])==0)){
         return(c(rtn.list,list(ok=FALSE, logLik = NaN,
         errors = paste("One of the diagonal elements of Sigma[,,",t,"]=0. That should never happen when t>1.  \nAre Q[i,i] and R[i,i] set to 0?\n",sep=""))))
      }
      if( any(takediag(Ft[,,t])==0) ){ #via check above this can only happen if corresponding t=1 and R & V0 corr elems = 0
        OmgF1=makediag(1,n)[takediag(Ft[,,t])!=0,,drop=FALSE] #permutation matrix
        if(dim(OmgF1)[1]==0){ #no non-zero Ft[,,1]
          detFt=1 #means R and diag(Ft[,,1] all 0; will become 0 when logged
          Ftinv=matrix(0,n,n)
        }else{
          #when R(i,i) is 0 then vt_t(i) will be zero and Sigma[i,i,1] will be 0 if V0=0.
          #OmgF1 makes sure we don't try to take 1/0 in
          if(length(OmgF1%*%Ft[,,t]%*%t(OmgF1))==1) detFt = OmgF1%*%Ft[,,t]%*%t(OmgF1) else detFt = det(OmgF1%*%Ft[,,t]%*%t(OmgF1))
          Ftinv = t(OmgF1)%*%chol2inv(chol(OmgF1%*%Ft[,,t]%*%t(OmgF1)))%*%OmgF1 #0s on row and col where 0 on diag
          }
      }else{
       if(n==1){ detFt=Ft[,,t]; Ftinv = 1/Ft[,,t] 
       }else{ 
         detFt=det(Ft[,,t]);
         Ftinv = chol2inv(chol(Ft[,,t]))
         Ftinv = (Ftinv + matrix(Ftinv,n,n,byrow=TRUE))/2 #enforce symmetry; Ft is symmetric; matrix call is faster than t()
         }
      }
      if( detFt<0 || !is.finite(log(detFt)) )
          return(c(rtn.list,list(ok=FALSE, logLik=NaN, Sigma=Ft, errors=paste("Stopped in MARSSkf: log(det(Ft[,,",t,"]))=NA.\n",sep="") ) ) )

      loglike = loglike - (1/2) %*% matrix(vt[,t],1,n) %*% Ftinv %*% vt[,t,drop=FALSE] - (1/2)*log(detFt)
      loglike = as.vector(loglike)
    }
    if( !is.finite(loglike) ) return(c(rtn.list,list(ok=FALSE, errors=paste("Stopped in MARSSkf: loglike computed to NA.\n")) ) )

    return(c(rtn.list,list(logLik = loglike, ok=TRUE, errors = msg)))
}

symm = function(x){
t.x = matrix(x,dim(x)[2],dim(x)[1],byrow=TRUE)
x=(x+t.x)/2
x
}
