#' Show historical changes to selected three-digit ICD-10-GM codes
#'
#' Show all changes in ICD history relating to the
#' 3-digit codes contained in a given vector icd
#'
#' @param icd3 Vector of three-digit ICD codes
#' @return data.frame with columns YEAR, ICD_CODE, ICD_LABEL and, if specified, DIAG_GROUP
#' @seealso [icd_showchanges()] if the input has been generated by [icd_expand()]
#' @examples
#' icd_showchanges_icd3("A09")
#' @export
icd_showchanges_icd3 <- function(icd3){
	out <- get_icd_history(icd3 = icd3)
	out[out$change == TRUE, ]
}

#' Show historical changes to selected ICD-10-GM codes
#'
#' Show all changes in ICD history relating to the
#' 3-digit codes contained in the data.frame icd_in.
#' The output of icd_expand can be passed directly to this
#' function to display relevant changes.
#'
#' @param icd_in Data frame created by  [icd_expand()], containing codes to check
#' @param years Years to check, corresponding to the variable `year_from` in `icd_meta_transition`
#' @return data.frame, as `icd_meta_transition`, with labels icd_from_label and icd_to_label
#' @seealso [icd_showchanges_icd3()] to provide one or more three-digit codes as input
#' @examples
#' dat_icd <- icd_expand(
#'    data.frame(ICD_SPEC = c("K52.9")),
#'    col_icd = "ICD_SPEC",
#'    year = 2019)
#' icd_showchanges(dat_icd)
#'
#' @importFrom rlang .data
#' @export
icd_showchanges <- function(icd_in, years=NULL) {

  if (is.null(years)) {
    years <- min(ICD10gm::icd_meta_transition$year_from):max(ICD10gm::icd_meta_transition$year_from)
  }

  ICD10gm::icd_meta_transition %>%
    # Only interested in codes that have changed
    dplyr::filter(.data$change) %>%
    # Restrict to the years of interest
    dplyr::filter(.data$year_from %in% years) %>%
    # Restrict to the codes contained in dat_icd
    dplyr::semi_join(icd_in, by = c("icd_from" = "icd_normcode")) %>%
    # Add labels
    dplyr::inner_join(ICD10gm::icd_meta_codes %>%
                        dplyr::select(.data$icd_normcode, .data$year, .data$label) %>%
                        dplyr::rename(icd_from_label = .data$label),
                      by = c("icd_from" = "icd_normcode",
                             "year_from" = "year")) %>%
    dplyr::inner_join(
      ICD10gm::icd_meta_codes %>%
        dplyr::select(.data$icd_normcode, .data$year, .data$label) %>%
        dplyr::rename(icd_to_label = .data$label),
      by = c("icd_to" = "icd_normcode",
             "year_to" = "year")
    )
}
