\name{hhg.univariate.ks.combined.test}
\alias{hhg.univariate.ks.combined.test}


\title{The combined K-sample test statistics and their p-values}

\description{Function for performing the combined  omnibus distribution-free test of equality of distributions among K groups, as described in Heller et al. (2014). The test combines statistics from a range of partition sizes. }

\usage{
hhg.univariate.ks.combined.test(X,Y=NULL,NullTable=NULL,mmin=2,
  mmax=ifelse(is.null(Y),4,max(4,round(min(table(Y))/3))) ,
  aggregation.type='sum',score.type='LikelihoodRatio' ,combining.type='MinP',nr.perm=1000)
}

\arguments{
  \item{X}{A numeric vector of data values (tied observations are broken at random), or the test statistic as output from \code{\link{hhg.univariate.ks.stat}}.}
  \item{Y}{for \code{k} groups, a vector of integers with values \code{0:(k-1)} which specify the group each observation belongs to. Leave as Null if the input to \code{X} is the test statistic.}
  \item{NullTable}{The null table of the statistic, which can be downloaded from the software website or computed by the function \code{\link{hhg.univariate.ks.nulltable}}.}
  \item{mmin}{The minimum partition size of the ranked observations, default value is 2. Ignored if \code{NullTable} is non-null.}
  \item{mmax}{The maximum partition size of the ranked observations, default value is 1/3 the number of observations in the smallest group. Ignored if \code{NullTable} is non-null. }
  \item{aggregation.type}{a character string specifying the aggregation type, must be one of \code{"sum"} (default), or \code{"max"}. Ignored if \code{NullTable} is non-null  or \code{X} is the test stiatistic.} 
  \item{score.type}{a character string specifying the score type, must be one of \code{"LikelihoodRatio"} (default), or \code{"Pearson"}. Ignored if \code{NullTable} is non-null  or \code{X} is the test stiatistic. }
  \item{combining.type}{a character string specifying the combining type, must be one of \code{"MinP"} (default), \code{"Fisher"}, or \code{"both"}.}
  \item{nr.perm}{The number of permutations for the null distribution. Ignored if \code{NullTable} is non-null.}
}

\details{
  The test statistic and p-value of the recommended test for the K-sample problem in Heller et al. (2014). The default combining type in the minimum p-value, so the test statistic is the minimum p-value over the range of partition sizes m from \code{mmin} to \code{mmax}, where the p-value for a fixed partition size m is defined by the aggregation type and score type . 
}

\value{
Returns a \code{UnivariateStatistic} class object, with the following entries:

  \item{MinP}{The test statistic when the combining type is \code{"MinP"}.}
  
  \item{MinP.pvalue}{The p-value when the combining type is \code{"MinP"}.}
  
  \item{MinP.m.chosen}{The partition size m for which the p-value was the smallest.}
  
  \item{Fisher}{The test statistic when the combining type is \code{"Fisher"}.}
  
  \item{Fisher.pvalue}{The p-value when the combining type is \code{"Fisher"}.}
  
  \item{m.stats}{The statistic for each m in the range \code{mmin} to \code{mmax}.}
  
  \item{pvalues.of.single.m}{The p-values for each m in the range \code{mmin} to \code{mmax}.}
  
  \item{generated_null_table}{The null table object. Null if \code{NullTable} is non-null.}
  
  \item{stat.type}{"KSample-Combined"}
  
  \item{aggregation.type}{a character string specifying the aggregation type used in the , one of \code{"sum"} or \code{"max"}.}
  
  \item{score.type}{a character string specifying the score typeused in the test, one of \code{"LikelihoodRatio"} or \code{"Pearson"}. }
  
  \item{mmax}{The maximum partition size of the ranked observations used for MinP or Fisher test statistic.}
  
  \item{mmin}{The minimum partition size of the ranked observations used for MinP or Fisher test statistic.}
  
}

\references{
Heller, R., Heller, Y., Kaufman S., Brill B, & Gorfine, M. (2014). Consistent distribution-free K-sample and independence tests for univariate random variables \emph{arXiv:1410.6758}.
}

\author{
  Barak Brill and Shachar Kaufman.
}



\examples{
\dontrun{
#Two groups, each from a different normal mixture:
N0=30
N1=30
X = c(c(rnorm(N0/2,-2,0.7),rnorm(N0/2,2,0.7)),c(rnorm(N1/2,-1.5,0.5),rnorm(N1/2,1.5,0.5)))
Y = (c(rep(0,N0),rep(1,N1)))
plot(Y,X)

#I) Perform MinP & Fisher Tests - without existing null tables.
#Null tables are generated by the test function.

results = hhg.univariate.ks.combined.test(X,Y,nr.perm = 100)
results


#The null table can then be accessed.
generated.null.table = results$generated_null_table


#II)Perform MinP & Fisher Tests - with existing null tables. 

#null table for aggregation by summation: 
sum.nulltable = hhg.univariate.ks.nulltable(c(N0,N1), nr.replicates=1000) 

MinP.Sm.existing.null.table = hhg.univariate.ks.combined.test(X,Y, NullTable = sum.nulltable)

#Results
MinP.Sm.existing.null.table

# combined test can also be performed by using the test statistic.
Sm.statistic = hhg.univariate.ks.stat(X,Y)
MinP.using.statistic = hhg.univariate.ks.combined.test(Sm.statistic, NullTable = sum.nulltable)
# same result as above
MinP.using.statistic$MinP.pvalue

#null table for aggregation by maximization: 
max.nulltable = hhg.univariate.ks.nulltable(c(N0,N1), aggregation.type = 'max', 
  score.type='LikelihoodRatio', mmin = 2, mmax = 10, nr.replicates = 100)

#combined test using both "MinP" and "Fisher":
MinPFisher.Mm.result = hhg.univariate.ks.combined.test(X,Y,NullTable =  max.nulltable ,
  combining.type = 'Both')
MinPFisher.Mm.result


}


}
