% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_infections.R
\name{estimate_infections}
\alias{estimate_infections}
\title{Estimate Infections, the Time-Varying Reproduction Number and the Rate of Growth}
\usage{
estimate_infections(
  reported_cases,
  generation_time,
  delays = delay_opts(),
  truncation = trunc_opts(),
  rt = rt_opts(),
  backcalc = backcalc_opts(),
  gp = gp_opts(),
  obs = obs_opts(),
  stan = stan_opts(),
  horizon = 7,
  CrIs = c(0.2, 0.5, 0.9),
  id = "estimate_infections",
  verbose = interactive()
)
}
\arguments{
\item{reported_cases}{A data frame of confirmed cases (confirm) by date (date). confirm must be integer and date must be
in date format.}

\item{generation_time}{A list containing the mean, standard deviation of the mean (mean_sd),
standard deviation (sd), standard deviation of the standard deviation and the maximum allowed value for the
generation time (assuming a gamma distribution).}

\item{delays}{A call to \code{delay_opts()} defining delay distributions and options. See the documentation of \code{delay_opts()}
and the examples below for details.}

\item{truncation}{\ifelse{html}{\out{<a href='https://www.tidyverse.org/lifecycle/#experimental'><img src='figures/lifecycle-experimental.svg' alt='Experimental lifecycle'></a>}}{\strong{Experimental}} A list of options as generated by \code{trunc_opts()}
defining the truncation of observed data. Defaults to \code{trunc_opts()}. See \code{estimate_truncation()}
for an approach to estimating truncation from data.}

\item{rt}{A list of options as generated by \code{rt_opts()} defining Rt estimation.
Defaults to \code{rt_opts()}. Set to \code{NULL} to switch to using  back calculation
rather than generating infections using Rt.}

\item{backcalc}{A list of options as generated by \code{backcalc_opts()} to define the
back calculation. Defaults to \code{backcalc_opts()}.}

\item{gp}{A list of options as generated by \code{gp_opts()} to define the
Gaussian process. Defaults to \code{gp_opts()}.Set to NULL to disable the
Gaussian process.}

\item{obs}{A list of options as generated by \code{obs_opts()} defining the
observation model. Defaults to \code{obs_opts()}.}

\item{stan}{A list of stan options as generated by \code{stan_opts()}. Defaults to \code{stan_opts()}. Can be used to override
\code{data}, \code{init}, and \code{verbose} settings if desired.}

\item{horizon}{Numeric, defaults to 7. Number of days into the future to forecast.}

\item{CrIs}{Numeric vector of credible intervals to calculate.}

\item{id}{A character string used to assign logging information on error. Used by \code{regional_epinow}
to assign errors to regions. Alter the default to run with error catching.}

\item{verbose}{Logical, defaults to \code{TRUE} when used interactively and otherwise \code{FALSE}. Should verbose debug progress messages be printed. Corresponds to the "DEBUG" level from
\code{futile.logger}. See \code{setup_logging} for more detailed logging options.}
}
\description{
\ifelse{html}{\out{<a href='https://www.tidyverse.org/lifecycle/#maturing'><img src='figures/lifecycle-maturing.svg' alt='Maturing lifecycle'></a>}}{\strong{Maturing}}
Uses a non-parametric approach to reconstruct cases by date of infection from reported
cases. It uses either a generative Rt model or non-parametric back calculation to estimate underlying
latent infections and then maps these infections to observed cases via uncertain reporting delays and a flexible
observation model. See the examples and function arguments for the details of all options. The default settings
may not be sufficient for your use case so the number of warmup samples (\code{stan_args = list(warmup)}) may need to
be increased as may the overall number of samples. Follow the links provided by any warnings messages to diagnose
issues with the MCMC fit. It is recommended to explore several of the Rt estimation approaches supported as not all
of them may be suited to users own use cases. See \href{https://gist.github.com/seabbs/163d0f195892cde685c70473e1f5e867}{here}
for an example of using \code{estimate_infections} within the \code{epinow} wrapper to estimate Rt for Covid-19 in a country from
the ECDC data source.
}
\examples{
\donttest{
# set number of cores to use
options(mc.cores = ifelse(interactive(), 4, 1))
# get example case counts
reported_cases <- example_confirmed[1:60]

# set up example generation time
generation_time <- get_generation_time(disease = "SARS-CoV-2", source = "ganyani")
# set delays between infection and case report 
incubation_period <- get_incubation_period(disease = "SARS-CoV-2", source = "lauer")
reporting_delay <- list(mean = convert_to_logmean(3, 1), mean_sd = 0.1,
                        sd = convert_to_logsd(3, 1), sd_sd = 0.1, max = 10)
      
# default setting 
# here we assume that the observed data is truncated by the same delay as 
def <- estimate_infections(reported_cases, generation_time = generation_time,
                           delays = delay_opts(incubation_period, reporting_delay),
                           rt = rt_opts(prior = list(mean = 2, sd = 0.1)),
                           stan = stan_opts(control = list(adapt_delta = 0.95)))
# real time estimates
summary(def)
# summary plot
plot(def)

# decreasing the accuracy of the approximate Gaussian to speed up computation. 
# These settings are an area of active research. See ?gp_opts for details.
agp <- estimate_infections(reported_cases, generation_time = generation_time,
                           delays = delay_opts(incubation_period, reporting_delay),
                           rt = rt_opts(prior = list(mean = 2, sd = 0.1)),
                           gp = gp_opts(ls_min = 10, basis_prop = 0.1),
                           stan = stan_opts(control = list(adapt_delta = 0.95)))
summary(agp)
plot(agp) 

# Adjusting for future susceptible depletion
dep <- estimate_infections(reported_cases, generation_time = generation_time,
                           delays = delay_opts(incubation_period, reporting_delay),
                           rt = rt_opts(prior = list(mean = 2, sd = 0.1),
                                        pop = 1000000, future = "latest"),
                           gp = gp_opts(ls_min = 10, basis_prop = 0.1), horizon = 21,
                           stan = stan_opts(control = list(adapt_delta = 0.95)))
plot(dep) 

# Adjusting for truncation of the most recent data
# See estimate_truncation for an approach to estimating this from data
trunc_dist <- list(mean = convert_to_logmean(0.5, 0.5), mean_sd = 0.1,
                   sd = convert_to_logsd(0.5, 0.5), sd_sd = 0.1, 
                   max = 3)
trunc <- estimate_infections(reported_cases, generation_time = generation_time,
                             delays = delay_opts(incubation_period, reporting_delay),
                             truncation = trunc_opts(trunc_dist),
                             rt = rt_opts(prior = list(mean = 2, sd = 0.1)),
                             gp = gp_opts(ls_min = 10, basis_prop = 0.1),
                             stan = stan_opts(control = list(adapt_delta = 0.95)))
plot(trunc) 

# using back calculation (combined here with under reporting)
# this model is in the order of 10 ~ 100 faster than the gaussian process method
# it is likely robust for retrospective Rt but less reliable for real time estimates
# the width of the prior window controls the reliance on observed data and can be 
# optionally switched off using backcalc_opts(prior = "none"), see ?backcalc_opts for 
# other options
backcalc <- estimate_infections(reported_cases, generation_time = generation_time,
                                delays = delay_opts(incubation_period, reporting_delay),
                                rt = NULL, backcalc = backcalc_opts(),
                                obs = obs_opts(scale = list(mean = 0.4, sd = 0.05)),
                                horizon = 0)
plot(backcalc)
                           
# Rt projected into the future using the Gaussian process
project_rt <- estimate_infections(reported_cases, generation_time = generation_time,
                                  delays = delay_opts(incubation_period, reporting_delay),
                                  rt = rt_opts(prior = list(mean = 2, sd = 0.1), 
                                               future = "project"))
plot(project_rt)

# default settings on a later snapshot of data 
snapshot_cases <- example_confirmed[80:130]
snapshot <- estimate_infections(snapshot_cases, generation_time = generation_time,
                                delays = delay_opts(incubation_period, reporting_delay),
                                rt = rt_opts(prior = list(mean = 1, sd = 0.1)))
plot(snapshot) 

# stationary Rt assumption (likely to provide biased real-time estimates)
stat <- estimate_infections(reported_cases, generation_time = generation_time,
                            delays = delay_opts(incubation_period, reporting_delay),
                            rt = rt_opts(prior = list(mean = 2, sd = 0.1), gp_on = "R0"))
plot(stat)
       
# no gaussian process (i.e fixed Rt assuming no breakpoints)
fixed <- estimate_infections(reported_cases, generation_time = generation_time,
                             delays = delay_opts(incubation_period, reporting_delay),
                             gp = NULL)
plot(fixed)

# no delays 
no_delay <- estimate_infections(reported_cases, generation_time = generation_time)
plot(no_delay)    

# break point but otherwise static Rt
bp_cases <- data.table::copy(reported_cases)
bp_cases <- bp_cases[, breakpoint := ifelse(date == as.Date("2020-03-16"), 1, 0)]
bkp <- estimate_infections(bp_cases, generation_time = generation_time,
                           delays = delay_opts(incubation_period, reporting_delay),
                           rt = rt_opts(prior = list(mean = 2, sd = 0.1)),
                           gp = NULL)                                                         
# break point effect
summary(bkp, type = "parameters", params = "breakpoints")
plot(bkp)

# weekly random walk
rw <- estimate_infections(reported_cases, generation_time = generation_time,
                          delays = delay_opts(incubation_period, reporting_delay),
                          rt = rt_opts(prior = list(mean = 2, sd = 0.1), rw = 7),
                          gp = NULL)     

# random walk effects
summary(rw, type = "parameters", params = "breakpoints") 
plot(rw)
}                                
}
\seealso{
epinow regional_epinow forecast_infections simulate_infections
}
