% Copyright 2013 Stefan Widgren and Maria Noremark,
% National Veterinary Institute, Sweden
%
% Licensed under the EUPL, Version 1.1 or - as soon they
% will be approved by the European Commission - subsequent
% versions of the EUPL (the "Licence");
% You may not use this work except in compliance with the
% Licence.
% You may obtain a copy of the Licence at:
%
% http://ec.europa.eu/idabc/eupl
%
% Unless required by applicable law or agreed to in
% writing, software distributed under the Licence is
% distributed on an "AS IS" basis,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
% express or implied.
% See the Licence for the specific language governing
% permissions and limitations under the Licence.

\name{Trace}
\alias{Trace}
\title{Trace Contacts.}
\usage{
  Trace(movements, root, tEnd, days, inBegin, inEnd,
    outBegin, outEnd)
}
\arguments{
  \item{movements}{a \code{data.frame} data.frame with
  movements, see details.}

  \item{root}{vector of roots to perform contact tracing
  for.}

  \item{tEnd}{the last date to include ingoing and outgoing
  movements. Defaults to \code{NULL}}

  \item{days}{the number of previous days before tEnd to
  include ingoing and outgoing movements. Defaults to
  \code{NULL}}

  \item{inBegin}{the first date to include ingoing
  movements. Defaults to \code{NULL}}

  \item{inEnd}{the last date to include ingoing movements.
  Defaults to \code{NULL}}

  \item{outBegin}{the first date to include outgoing
  movements. Defaults to \code{NULL}}

  \item{outEnd}{the last date to include outgoing
  movements. Defaults to \code{NULL}}
}
\description{
  Contact tracing for a specied node(s) (root) during a
  specfied time period. The time period is divided into two
  parts, one for ingoing contacts and one for outgoing
  contacts.
}
\details{
  The time period used for \code{Trace} can either be
  specified using \code{tEnd} and \code{days} or
  \code{inBegin}, \code{inEnd}, \code{outBegin} and
  \code{outEnd}.

  If using \code{tEnd} and \code{days}, the time period for
  ingoing and outgoing contacts ends at \code{tEnd} and
  starts at \code{days} prior to \code{tEnd}. The tracing
  will be performed for each combination of \code{root},
  \code{tEnd} and \code{days}.

  An alternative way is to use \code{inBegin},
  \code{inEnd}, \code{outBegin} and \code{outEnd}. The time
  period for ingoing contacts starts at inBegin and ends at
  inEndDate.  For outgoing contacts the time period starts
  at outBegin and ends at outEnd. The vectors \code{root}
  \code{inBegin}, \code{inEnd}, \code{outBegin} and
  \code{outEnd} must have the same lengths and the tracing
  will be performed for each index of them.

  The argument movements in Trace is a \code{data.frame}
  with the following columns: \describe{

  \item{source}{ an integer or character identifier of the
  source holding.  }

  \item{destination}{ an integer or character identifier of
  the destination holding.  }

  \item{t}{ the Date of the transfer }

  \item{id}{ an optional character vector with the identity
  of the animal.  }

  \item{n}{ an optional numeric vector with the number of
  animals moved.  }

  \item{category}{ an optional character or factor with
  category of the animal e.g. Cattle.  } }
}
\examples{
## Load data
data(transfers)

## Perform contact tracing using tEnd and days
trace.1 <- Trace(movements=transfers,
                 root=2645,
                 tEnd='2005-10-31',
                 days=91)

## Perform contact tracing using inBegin, inEnd
## outBegin and outEnd
trace.2 <- Trace(movements=transfers,
                 root=2645,
                 inBegin='2005-08-01',
                 inEnd='2005-10-31',
                 outBegin='2005-08-01',
                 outEnd='2005-10-31')

## Check that the result is identical
identical(trace.1, trace.2)

## Show result of contact tracing
show(trace.1)

\dontrun{
## Plot in- and outgoing contact chain
plot(trace.1)

## Create a network summary for all included herds
## First extract all source and destination from the dataset
root <- sort(unique(c(transfers$source,
                      transfers$destination)))

## Perform contact tracing using tEnd and days.
trace.3 <- Trace(movements=transfers,
                 root=root,
                 tEnd='2005-10-31',
                 days=91)

## Perform contact tracing using inBegin, inEnd
## outBegin and outEnd
trace.4 <- Trace(movements=transfers,
                 root=root,
                 inBegin=rep('2005-08-01', length(root)),
                 inEnd=rep('2005-10-31', length(root)),
                 outBegin=rep('2005-08-01', length(root)),
                 outEnd=rep('2005-10-31', length(root)))

## Check that the result is identical
identical(trace.3, trace.4)

NetworkSummary(trace.3)
}
}
\references{
  \itemize{ \item Dube, C., et al., A review of network
  analysis terminology and its application to
  foot-and-mouth disease modelling and policy development.
  Transbound Emerg Dis 56 (2009) 73-85, doi:
  10.1111/j.1865-1682.2008.01064.x

  \item Noremark, M., et al., Network analysis of cattle
  and pig movements in Sweden: Measures relevant for
  disease control and riskbased surveillance.  Preventive
  Veterinary Medicine 99 (2011) 78-90, doi:
  10.1016/j.prevetmed.2010.12.009 }
}
\seealso{
  \code{\link{Trace}}.
}

