% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mk_cond.R
\name{mk_cond}
\alias{mk_cond}
\title{Helper for making conditions for some functions}
\usage{
mk_cond(...)
}
\arguments{
\item{...}{sets of 3 elements for each condition. Each
set of three elements is the: (1) node or edge attribute
name (character value), (2) the conditional operator
(character value), and (3) the non-attribute operand. If
there are multiple conditions to be specified, then a
\code{&} or \code{|} operator must be used between each
condition, specifying an \code{AND} or \code{OR}.}
}
\value{
a string to be used for any \code{conditions}
argument.
}
\description{
Create one or multiple conditions for
all traversal functions (\code{trav_...}) and
certain selection functions (\code{select_nodes()} and
\code{select_edges()}). This helper could be invoked
for these functions' \code{conditions} argument.
}
\examples{
# Create a node data frame (ndf)
ndf <-
  create_node_df(
    n = 4,
    type = "basic",
    label = TRUE,
    value = c(3.5, 2.6, 9.4, 2.7))

# Create an edge data frame (edf)
edf <-
  create_edge_df(
    from = c(1, 2, 3),
    to = c(4, 3, 1),
    rel = c("z", "z", "a"),
    value = c(6.4, 2.9, 5.0))

# Create a graph with the ndf and edf
graph <-
  create_graph(
    nodes_df = ndf,
    edges_df = edf)

# Select edges where the `rel` label is `z` using
# the `mk_cond()` helper function
graph_1 <-
  graph \%>\%
  select_edges(
    conditions =
      mk_cond(
        "rel",  "==",  "z"))

# Verify that an edge selection has been made; the
# edges corresponding to this condition are the
# `1->4` and 2->3` edges with edge IDs `1` and `2`
get_selection(graph_1)
#> [1] 1 2

# Select edges based on the relationship label
# being `z` and the `value` being < 5.0
graph_2 <-
  graph \%>\%
  select_edges(
    mk_cond(
      "rel",  "==",  "z",
      "&",
      "value", "<", 5.0))

# Verify that an edge selection has been made; the
# edge corresponding to these conditions is the
# `2->3` edge with ID `2`
get_selection(graph_2)
#> [1] 2

# We can mix condition strings and conditions made
# with the `mk_cond()` helper function
# being `z` and the `value` being < 5.0
graph_2b <-
  graph \%>\%
  select_edges(
    conditions =
    c("rel == 'z'",
      mk_cond("value", "<", 5.0)))

# This selection will be the same as that previous;
# note that conditions collected as a vector with
# `c()` are AND conditions
get_selection(graph_2) == get_selection(graph_2b)
#> [1] TRUE

# Because we are not specifying conditions as
# single strings, we can use objects from the
# workspace (or function calls) to compose the
# condition(s)

# Create the `rel_select` character vector
rel_select <- "a"

# Use a condition that gets the `rel` operand
# from an object, and, a `value` operand that
# is calculated from the mean of its values
# in the graph's edge data frame (~4.77)
graph_3 <-
  graph \%>\%
  select_edges(
    mk_cond(
      "rel",  "==", rel_select,
      "&",
      "value", ">", get_edge_attrs(., "value") \%>\%
                    mean()))

# Verify that an edge selection has been made; the
# edge corresponding to these conditions is the
# `3->1` edge with ID `3`
get_selection(graph_3)
#> [1] 3
}

