% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Ternary.R
\name{ternary_data}
\alias{ternary_data}
\title{Prepare data for showing contours in ternary diagrams.}
\usage{
ternary_data(
  prop = c(".P1", ".P2", ".P3"),
  add_var = list(),
  resolution = 3,
  prediction = TRUE,
  ...
)
}
\arguments{
\item{prop}{A character vector specifying the columns names of compositional
variables whose proportions to manipulate. Default is ".P1", ".P2",
and ".P3".}

\item{add_var}{A list or data-frame specifying values for additional variables
in the model other than the proportions (i.e. not part of the
simplex design).
This could be useful for comparing the predictions across
different values for a non-compositional variable.
If specified as a list, it will be expanded to show a plot
for each unique combination of values specified, while if specified
as a data-frame, one plot would be generated for each row in the
data.}

\item{resolution}{A number between 1 and 10 describing the resolution of the
resultant graph.
A high value would result in a higher definition figure
but at the cost of being computationally expensive.}

\item{prediction}{A logical value indicating whether to pass the final data
to the `\link{add_prediction}` function and append the
predictions to the data. Default value is \code{TRUE}, but
often it would be desirable to make additional changes to
the data before making any predictions, so the user can set this to
\code{FALSE} and manually call the `\link{add_prediction}`
function.}

\item{...}{
  Arguments passed on to \code{\link[=add_prediction]{add_prediction}}
  \describe{
    \item{\code{model}}{A regression model object which will be used to make predictions
for the observations in `data`. Will override `coefficients`
if specified.}
    \item{\code{coefficients}}{If a regression model is not available (or can't be fit in R),
the regression coefficients from a model fit in some other
language can be used to calculate predictions. However, the
user would have to ensure there's an appropriate one-to-one
positional mapping between the data columns and the
coefficient values. Further, they would also have to provide
a variance-covariance matrix of the coefficients in the `vcov`
parameter if they want the associated CI for the prediction or
it would not be possible to calculate confidence/prediction
intervals using this method.}
    \item{\code{vcov}}{If regression coefficients are specified, then the variance-covariance
matrix of the coefficients can be specified here to calculate the
associated confidence interval around each prediction. Failure to do
so would result in no confidence intervals being returned. Ensure
`coefficients` and `vcov` have the same positional mapping with the data.}
    \item{\code{coeff_cols}}{If `coefficients` are specified and a one-to-one positional
mapping between the data-columns and coefficient vector is
not present. A character string or numeric index can be specified
here to reorder the data columns and match the corresponding
coefficient value to the respective data column. See the
"Use model coefficients for prediction" section in examples.}
    \item{\code{conf.level}}{The confidence level for calculating confidence/prediction
intervals. Default is 0.95.}
    \item{\code{interval}}{Type of interval to calculate:
\describe{
  \item{"none" (default)}{No interval to be calculated.}
  \item{"confidence"}{Calculate a confidence interval.}
  \item{"prediction"}{Calculate a prediction interval.}
}}
  }}
}
\value{
A data-frame with the following columns and any additional columns
        specified in `add_var` parameter
 \describe{
   \item{.x}{The x component of the x-y projection of the simplex point.}
   \item{.y}{The y component of the x-y projection of the simplex point.}
   \item{.P1}{The first variable whose proportion is varied across the simplex.}
   \item{.P2}{The second variable whose proportion is varied across the simplex.}
   \item{.P3}{The third variable whose proportion is varied across the simplex.}
   \item{.add_str_ID}{An identifier column for grouping the cartesian product
                      of all additional columns specified in `add_var`
                      parameter (if `add_var` is specified).}
   \item{.Pred}{The predicted response for each observation
               (if `prediction` is \code{TRUE}).}
   \item{.Lower}{The lower limit of the prediction/confidence interval
                 for each observation.}
   \item{.Upper}{The upper limit of the prediction/confidence interval
                 for each observation.}
 }
}
\description{
The data preparation function for creating an equally spaced grid of three
compositional variables (i.e., the three variables sum to 1 at each point
along the grid). The projection of each point in the grid on the x-y plane is
also calculated. This data can be used with a relevant statistical model
to predict the response across the ternary surface. The output of this
function can then be passed to the \code{\link{ternary_plot}} function to
visualise the change in the response as a contour plot. \cr
\emph{Note:} This function works only for models with three compositional
predictors. For models with more than three compositional predictors see
\code{\link{conditional_ternary}}.
}
\examples{
library(DImodels)
library(dplyr)

## Load data
data(sim0)

## Fit model
mod <- lm(response ~ 0 + (p1 + p2 + p3)^2, data = sim0)

## Prepare data for creating a contour map of predicted response over
## the ternary surface
## Remember to specify prop with the same character values as the names
## of the variables in the model containing the prop.
plot_data <- ternary_data(resolution = 1, model = mod,
                          prop = c("p1", "p2", "p3"))
## Show plot
ternary_plot(data = plot_data)

## Can also add any additional variables independent of the simplex using
## the `add_var` argument
sim0$treatment <-  rep(c("A", "B", "C", "D"), each = 16)
new_mod <- update(mod, ~. + treatment, data = sim0)
plot_data <- ternary_data(prop = c("p1", "p2", "p3"),
                          add_var = list("treatment" = c("A", "B")),
                          resolution = 1, model = new_mod)
## Plot to compare between additional variables
\donttest{
ternary_plot(plot_data)
}

## It could be desirable to take the output of this function and add
## additional variables to the data before making predictions
## Use `prediction = FALSE` to get data without any predictions
contour_data <- ternary_data(prop = c("p1", "p2", "p3"),
                             model = mod,
                             prediction = FALSE,
                             resolution = 1)
head(contour_data)

## Manually add the treatment variable
contour_data$treatment <- "A"
## Make predictions
head(add_prediction(data = contour_data, model = new_mod))

## Manually add the interaction terms
contour_data <- contour_data \%>\%
                  mutate(`p1:p2` = p1*p2,
                         `p2:p3` = p2*p3,
                         `p1:p3` = p1*p3)

## Add predictions using model coefficients
contour_data <- add_prediction(data = contour_data,
                               coefficient = mod$coefficient)
head(contour_data)

## Note: Add predictions via coefficients would not give confidence intervals
## to get CIs using coefficients we need to specify the variance-covariance
## matrix using `vcov`
contour_data <- add_prediction(data = contour_data,
                               coefficient = mod$coefficient,
                               vcov = vcov(mod),
                               interval = "confidence")
head(contour_data)
## Show plot
\donttest{
ternary_plot(contour_data)
}
## See `?ternary_plot` for options to customise the ternary_plot
}
