% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/homophily.fe.R
\name{homophily.fe}
\alias{homophily.fe}
\title{Estimating Network Formation Models with Degree Heterogeneity: the Fixed Effect Approach}
\usage{
homophily.fe(
  network,
  formula,
  data,
  symmetry = FALSE,
  fe.way = 1,
  init = NULL,
  method = c("L-BFGS", "Block-NRaphson", "Mix"),
  ctr = list(maxit.opt = 10000, maxit.nr = 50, eps_f = 1e-09, eps_g = 1e-09, tol = 1e-04),
  print = TRUE
)
}
\arguments{
\item{network}{A matrix or list of sub-matrices of social interactions containing 0 and 1, where links are represented by 1.}

\item{formula}{An object of class \link[stats]{formula}: a symbolic description of the model. The \code{formula} should be, for example, \code{~ x1 + x2},
where \code{x1} and \code{x2} are explanatory variables for link formation. If missing, the model is estimated with fixed effects only.}

\item{data}{An optional data frame, list, or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in data, the variables are taken from \code{environment(formula)}, typically the environment from which \code{homophily} is called.}

\item{symmetry}{Indicates whether the network model is symmetric (see details).}

\item{fe.way}{Indicates whether it is a one-way or two-way fixed effect model. The expected value is 1 or 2 (see details).}

\item{init}{(optional) Either a list of starting values containing \code{beta}, a K-dimensional vector of the explanatory variables' parameters,
\code{mu}, an n-dimensional vector, and \code{nu}, an n-dimensional vector, where K is the number of explanatory variables and n is the number of individuals;
or a vector of starting values for \code{c(beta, mu, nu)}.}

\item{method}{A character string specifying the optimization method. Expected values are \code{"L-BFGS"}, \code{"Block-NRaphson"}, or \code{"Mix"}.
\code{"Block-NRaphson"} refers to the \code{Newton-Raphson} method applied to each subnetwork, and \code{"Mix"} combines the \code{Newton-Raphson} method for \code{beta} with the \code{L-BFGS} method for the fixed effects.}

\item{ctr}{(optional) A list containing control parameters for the solver. For the \code{optim_lbfgs} method from the \pkg{RcppNumerical} package,
the list should include \code{maxit.opt} (corresponding to \code{maxit} for the \code{L-BFGS} method), \code{eps_f}, and \code{eps_g}. For the \code{Block-NRaphson} method,
the list should include \code{maxit.nr} (corresponding to \code{maxit} for the \code{Newton-Raphson} method) and \code{tol}.}

\item{print}{A boolean indicating if the estimation progression should be printed.}
}
\value{
A list consisting of:
\item{model.info}{A list of model information, such as the type of fixed effects, whether the model is symmetric,
the number of observations, etc.}
\item{estimate}{The maximizer of the log-likelihood.}
\item{loglike}{The maximized log-likelihood.}
\item{optim}{The returned value from the optimization solver, which contains details of the optimization. The solver used is \code{optim_lbfgs} from the
\pkg{RcppNumerical} package.}
\item{init}{The returned list of starting values.}
\item{loglike.init}{The log-likelihood at the starting values.}
}
\description{
\code{homophily.fe} implements a Logit estimator for a network formation model with homophily. The model includes degree heterogeneity using fixed effects (see details).
}
\details{
Let \eqn{p_{ij}} be the probability for a link to go from individual \eqn{i} to individual \eqn{j}.
This probability is specified for two-way effect models (\code{fe.way = 2}) as
\deqn{p_{ij} = F(\mathbf{x}_{ij}'\beta + \mu_i + \nu_j),}
where \eqn{F} is the cumulative distribution function of the standard logistic distribution. Unobserved degree heterogeneity is captured by
\eqn{\mu_i} and \eqn{\nu_j}. These are treated as fixed effects (see \code{\link{homophily.re}} for random effect models).
As shown by Yan et al. (2019), the estimator of the parameter \eqn{\beta} is biased. A bias correction is necessary but not implemented in this version. However,
the estimators of \eqn{\mu_i} and \eqn{\nu_j} are consistent.\cr

For one-way fixed effect models (\code{fe.way = 1}), \eqn{\nu_j = \mu_j}. For symmetric models, the network is not directed, and the fixed effects need to be one-way.
}
\examples{
\donttest{
set.seed(1234)
M            <- 2 # Number of sub-groups
nvec         <- round(runif(M, 20, 50))
beta         <- c(.1, -.1)
Glist        <- list()
dX           <- matrix(0, 0, 2)
mu           <- list()
nu           <- list()
Emunu        <- runif(M, -1.5, 0) # Expectation of mu + nu
smu2         <- 0.2
snu2         <- 0.2
for (m in 1:M) {
  n          <- nvec[m]
  mum        <- rnorm(n, 0.7*Emunu[m], smu2)
  num        <- rnorm(n, 0.3*Emunu[m], snu2)
  X1         <- rnorm(n, 0, 1)
  X2         <- rbinom(n, 1, 0.2)
  Z1         <- matrix(0, n, n)  
  Z2         <- matrix(0, n, n)
  
  for (i in 1:n) {
    for (j in 1:n) {
      Z1[i, j] <- abs(X1[i] - X1[j])
      Z2[i, j] <- 1*(X2[i] == X2[j])
    }
  }
  
  Gm           <- 1*((Z1*beta[1] + Z2*beta[2] +
                       kronecker(mum, t(num), "+") + rlogis(n^2)) > 0)
  diag(Gm)     <- 0
  diag(Z1)     <- NA
  diag(Z2)     <- NA
  Z1           <- Z1[!is.na(Z1)]
  Z2           <- Z2[!is.na(Z2)]
  
  dX           <- rbind(dX, cbind(Z1, Z2))
  Glist[[m]]   <- Gm
  mu[[m]]      <- mum
  nu[[m]]      <- num
}

mu  <- unlist(mu)
nu  <- unlist(nu)

out   <- homophily.fe(network =  Glist, formula = ~ -1 + dX, fe.way = 2)
muhat <- out$estimate$mu
nuhat <- out$estimate$nu
plot(mu, muhat)
plot(nu, nuhat)
}
}
\references{
Yan, T., Jiang, B., Fienberg, S. E., & Leng, C. (2019). Statistical inference in a directed network model with covariates. \emph{Journal of the American Statistical Association}, 114(526), 857-868, \doi{https://doi.org/10.1080/01621459.2018.1448829}.
}
\seealso{
\code{\link{homophily.re}}.
}
