% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simBuyseTest.R
\name{Simulate endpoints for GPC}
\alias{Simulate endpoints for GPC}
\alias{simBuyseTest}
\title{Simulation of data for the BuyseTest}
\usage{
simBuyseTest(
  n.T,
  n.C = NULL,
  argsBin = list(),
  argsCont = list(),
  argsTTE = list(),
  n.strata = NULL,
  names.strata = NULL,
  format = "data.table",
  latent = FALSE
)
}
\arguments{
\item{n.T}{[integer, >0] number of patients in the treatment arm}

\item{n.C}{[integer, >0] number of patients in the control arm}

\item{argsBin}{[list] arguments to be passed to \code{simBuyseTest_bin}. They specify the distribution parameters of the categorical endpoints.}

\item{argsCont}{[list] arguments to be passed to \code{simBuyseTest_continuous}. They specify the distribution parameters of the continuous endpoints.}

\item{argsTTE}{[list]  arguments to be passed to \code{simBuyseTest_TTE}. They specify the distribution parameters of the time to event endpoints.}

\item{n.strata}{[integer, >0] number of strata. \code{NULL} indicates no strata.}

\item{names.strata}{[character vector] name of the strata variables. Must have same length as \code{n.strata}.}

\item{format}{[character] the format of the output. Can be \code{"data.table"}, \code{"data.frame"} or \code{"matrix"}.}

\item{latent}{[logical] If \code{TRUE} also export the latent variables (e.g. censoring times or event times).}
}
\description{
Simulate categorical, continuous or time to event endpoints, possibly along with a strata variable.
Categorical endpoints are simulated by thresholding a latent Gaussian variable (tobit model),
continuous endpoints are simulated using a Gaussian distribution,
and time to event endpoints are simulated using Weibull distributions for the event of interest, competing events, and censoring.
This function is built upon the \code{lvm} and \code{sim} functions from the lava package.
}
\details{
Endpoints are simulated independently of the strata variable and independently of each other,
with the exception of categorical endpoint and the time to event endpoints that can be correlated
by specifying a non-0 value for the \code{rho.T} and \code{rho.C} elements of the argument \code{argsBin}.

Arguments in the list \code{argsBin}:
\itemize{
    \item\code{p.T} list of probabilities for the values taken by each endpoint (categorical endpoint, treatment group). 
    \item\code{p.C} same as \code{p.T} but for the control group. 
    \item\code{rho.T} value of the regression coefficient between the underlying latent variable and the survival time. 
    \item\code{rho.C} same as \code{rho.T} but for the control group. 
    \item\code{name} names of the binary variables.
}

Arguments in the list \code{argsCont}:
    \itemize{
    \item\code{mu.T} expected value of each endpoint (continuous endpoint, treatment group). 
    \item\code{mu.C} same as \code{mu.C} but for the control group. 
    \item\code{sigma.T} standard deviation of the values of each endpoint (continuous endpoint, treatment group). 
    \item\code{sigma.C} same as \code{sigma.T} but for the control group. 
    \item\code{name} names of the continuous variables.
    }

Arguments in the list \code{argsTTE}:
    \itemize{
    \item\code{CR} should competing risks be simulated? 
    \item\code{scale.T,scale.C,scale.CR,scale.Censoring.T,scale.Censoring.C} scale parameter of the Weibull distribution for, respectively,
     the event of interest in the treatment group,
     the event of interest in the control group,
     the competing event in both groups,
     the censoring mechanism in the treatment group,
     the censoring mechanism in the control group
    \item\code{shape.T,shape.C,shape.CR,shape.Censoring.T,shape.Censoring.C} shape parameter of the Weibull distribution for, respectively,
     the event of interest in the treatment group,
     the event of interest in the control group,
     the competing event in both groups,
     the censoring mechanism in the treatment group,
     the censoring mechanism in the control group
    \item\code{name} names of the time to event variables. 
    \item\code{nameCensoring} names of the event type indicators. 
    }
}
\examples{
library(data.table)

n <- 1e2

#### by default ####
simBuyseTest(n)

## with a strata variable having 5 levels
simBuyseTest(n, n.strata = 5)
## with a strata variable named grade
simBuyseTest(n, n.strata = 5, names.strata = "grade")
## several strata variables
simBuyseTest(1e3, n.strata = c(2,4), names.strata = c("Gender","AgeCategory"))

#### only categorical endpoints ####
args <- list(p.T = list(c(low=0.1,moderate=0.5,high=0.4)))
dt.bin <- simBuyseTest(n, argsBin = args, argsCont = NULL, argsTTE = NULL)
table(dt.bin$toxicity)/NROW(dt.bin)

args <- list(p.T = list(c(low=0.1,moderate=0.5,high=0.4), c(0.1,0.9)))
dt.bin <- simBuyseTest(n, argsBin = args, argsCont = NULL, argsTTE = NULL)
table(dt.bin$toxicity1)/NROW(dt.bin)
table(dt.bin$toxicity2)/NROW(dt.bin)

#### only continuous endpoints ####
args <- list(mu.T = c(3:5/10), sigma.T = rep(1,3))
dt.cont <- simBuyseTest(n, argsBin = NULL, argsCont = args, argsTTE = NULL)
c(mean(dt.cont$score1), mean(dt.cont$score2), mean(dt.cont$score3))
c(sd(dt.cont$score1), sd(dt.cont$score2), sd(dt.cont$score3))

#### only TTE endpoints ####
args <- list(scale.T = c(3:5/10), scale.Censoring.T = rep(1,3))
dt.tte <- simBuyseTest(n, argsBin = NULL, argsCont = NULL, argsTTE = args)
1/c(sum(dt.tte$eventtime1)/sum(dt.tte$status1),
  sum(dt.tte$eventtime2)/sum(dt.tte$status2),
  sum(dt.tte$eventtime3)/sum(dt.tte$status3))
        
1/c(sum(dt.tte$eventtime1)/sum(dt.tte$status1==0),
  sum(dt.tte$eventtime2)/sum(dt.tte$status2==0),
  sum(dt.tte$eventtime3)/sum(dt.tte$status3==0))

#### correlated categorical / time to event endpoint ####
args.bin <- list(p.T = list(c(low=0.1,moderate=0.5,high=0.4)), rho.T = 1)
args.tte <- list(scale.T = 2, scale.Censoring.T = 1)
dt.corr <- simBuyseTest(n, argsBin = args.bin, argsCont = NULL, argsTTE = args.tte)

1/(sum(dt.corr$eventtime)/sum(dt.corr$status))
1/(sum(dt.corr$eventtime)/sum(dt.corr$status==0))
table(dt.corr$toxicity)/NROW(dt.corr)

boxplot(eventtime ~ toxicity, data = dt.corr)

}
\author{
Brice Ozenne
}
\keyword{function}
\keyword{simulations}
