% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/befa.R
\name{befa}
\alias{befa}
\title{Bayesian Exploratory Factor Analysis}
\usage{
befa(model, data, burnin = 1000, iter = 10000, Nid = 3, Kmax, A0 = 10,
  B0 = 10, c0 = 2, C0 = 1, HW.prior = TRUE, nu0 = Kmax + 1, S0 = 1,
  kappa0 = 2, xi0 = 1, kappa = 1/Kmax, indp.tau0 = TRUE,
  rnd.step = TRUE, n.step = 5, search.delay = min(burnin, 10),
  R.delay = min(burnin, 100), dedic.start, loading.start, idiovar.start,
  beta.start, R.start, verbose = FALSE)
}
\arguments{
\item{model}{This argument can be specified in two different ways:
\itemize{
\item A numeric matrix or a data frame containing the manifest
  variables. This corresponds to a model without covariates,
  and the argument \code{data} is not required.
\item A list of model formulas. Each element of the list specifies
  a manifest variable and its corresponding control variables (e.g.,
  '\code{Y1 ~ X1 + X2}' to use \code{X1} and \code{X2} as control
  variables for \code{Y1}).
  If a formula has no left-hand side variable, covariates on the
  right-hand side are included in all equations (e.g., '\code{~ X3}'
  means that \code{X3} is used as a control variable for all the
  manifest variables). Argument \code{data} can be passed to the
  function in that case, otherwise parent data frame is used.
}
Binary variables should be specified as logical vectors in the data
frame to be treated as dichotomous. \code{NA} values are accepted in
manifest variables only.}

\item{data}{Data frame. If missing, parent data frame if used.}

\item{burnin}{Burn-in period of the MCMC sampler.}

\item{iter}{Number of MCMC iterations saved for posterior inference (after
burn-in).}

\item{Nid}{Minimum number of manifest variables dedicated to each latent factor
for identification.}

\item{Kmax}{Maximum number of latent factors. If missing, the maximum number of
factors that satisfies the identification condition determined by
\code{Nid} and the Ledermann bound is specified (see CFSHP,
section 2.2).}

\item{A0}{Scaling parameters of the variance of the Normal prior on the nonzero
factor loadings. Either a scalar or a numeric vector of length equal
to the number of manifest variables.}

\item{B0}{Variances of the Normal prior on the regression coefficients. Either a
scalar or a numeric vector of length equal to the number of
manifest variables.}

\item{c0}{Shape parameters of the Inverse-Gamma prior on the idiosyncratic
variances. Either a scalar or a numeric vector of length equal to the
number of manifest variables.}

\item{C0}{Scale parameters of the Inverse-Gamma prior on the idiosyncratic
variances. Either a scalar or a numeric vector of length equal to the
number of manifest variables.}

\item{HW.prior}{If \code{TRUE}, implement Huang-Wand (2013) prior on the covariance
matrix of the factors in the expanded model, otherwise use an
Inverse-Wishart prior if \code{FALSE}, see CFSHP section 2.3.5.}

\item{nu0}{Degrees of freedom of the Inverse-Wishart prior on the covariance
matrix of the latent factors in the expanded model.}

\item{S0}{Scale parameters of the Inverse-Wishart prior on the covariance matrix
of latent factors in the expanded model:
\itemize{
  \item if \code{HW.prior = TRUE}, scale parameter of the Gamma
  hyperprior distribution on the individual scales of the
  Inverse-Wishart prior.
  \item if \code{HW.prior = FALSE}, diagonal elements of the scale
  matrix of the Inverse-Wishart prior on the covariance matrix of
  the latent factors in the expanded model.
}
Either a scalar or a numeric vector of length equal to \code{Kmax}.}

\item{kappa0}{First shape parameter of the Beta prior distribution on the
probability \eqn{\tau_0} that a manifest variable does not load on any
factor.}

\item{xi0}{Second shape parameter of the Beta prior distribution on the
probability \eqn{\tau_0} that a manifest variable does not load on any
factor.}

\item{kappa}{Concentration parameters of the Dirichlet prior distribution on the
indicators. Either a scalar or a numeric vector of length equal to
\code{Kmax}.}

\item{indp.tau0}{If \code{TRUE}, specify the alternative prior specification with
independent parameters \eqn{\tau_{0m}}{\tau_0m} across manifest
variables \eqn{m = 1, ..., M}, otherwise use a common parameter
\eqn{\tau_0} if \code{FALSE}.}

\item{rnd.step}{If \code{TRUE}, select randomly the number of intermediate steps in
non-identified models at each MCMC iteration, otherwise use a fixed
number of steps if \code{FALSE}.}

\item{n.step}{Controls the number of intermediate steps in non-identified models:
\itemize{
  \item if \code{rnd.step = TRUE}, average number of steps. The number
  of steps is sampled at each MCMC iteration from
  1+Poisson(\code{n.step}-1).
  \item if \code{rnd.step = FALSE}, fixed number of steps.
}}

\item{search.delay}{Number of MCMC iterations run with fixed indicator matrix (specified
in \code{dedic.start}) at beginning of MCMC sampling.}

\item{R.delay}{Number of MCMC iterations run with fixed correlation matrix (specified
in \code{dedic.start}) at beginning of MCMC sampling.}

\item{dedic.start}{Starting values for the indicators. Vector of integers of length equal
to the number of manifest variables. Each element takes a value among
0, 1, ..., \code{Kmax}. If missing, random allocation of the manifest
variables to the maximum number of factors \code{Kmax}, with a minimum
of \code{Nid} manifest variables loading on each factor.}

\item{loading.start}{Starting values for the factor loadings. Numeric vector of length
equal to the number of manifest variables. If missing, sampled from a
Normal distribution with zero mean and variance \code{A0}.}

\item{idiovar.start}{Starting values for the idiosyncratic variances. Numeric vector of
length equal to the number of manifest variables. Sampled from prior
if missing.}

\item{beta.start}{Starting values for the regression coefficients. Numeric vector of
length equal to the total number of regression coefficients,
concatenated for all the manifest variables. Sampled from prior if
missing.}

\item{R.start}{Starting values for the correlation matrix of the latent factors.
Numeric matrix with \code{Kmax} rows and columns, and unit diagonal
elements. If missing, identity matrix is used.}

\item{verbose}{If \code{TRUE}, display information such as the progression of the
MCMC sampler.}
}
\value{
The function returns an object of class '\code{befa}' containing:
\itemize{
  \item \code{draws}: A matrix containing the MCMC draws of the model
  parameters in its columns, stored in the following order: factor loadings,
  idiosyncratic variances, regression coefficients (if any), off-diagonal
  elements of the correlation matrix of the factors.
  \item \code{dedic}: A matrix containing the MCMC draws of the indicators.
  \item \code{MHacc}: Acceptance rate of the Metropolis-Hastings algorithm.
}
The parameters \code{Kmax} and \code{Nid} are saved as object attributes.

Note that identification is achieved only with respect to the scale of the
latent factors. Non-identifiability problems may affect the posterior sample
because of column switching and sign switching of the factor loadings.
These issues can be addressed \emph{a posteriori} with the functions
\code{\link{post.column.switch}} and \code{\link{post.sign.switch}}.
}
\description{
This function implements the Bayesian Exploratory Factor Analysis
(\code{befa}) developed in Conti et al. (CFSHP, 2014). It runs a MCMC sampler
for a factor model with dedicated factors, where each manifest variable is
allowed to load on at most one latent factor. The allocation of the manifest
variables to the latent factors is not fixed \emph{a priori} but determined
stochastically during sampling. The minimum number of variables dedicated to
each factor can be controlled by the user to achieve the desired level of
identification. The manifest variables can be continuous or dichotomous, and
control variables can be introduced as covariates.
}
\details{
\strong{Model specification.} The model is specified as follows, for
each observation \eqn{i = 1, ..., N}:
  \deqn{Y^*_i = \beta X_i + \alpha \theta_i + \epsilon_i}{
        Y*_i = \beta X_i + \alpha \theta_i + \epsilon_i}
  \deqn{\theta_i \sim \mathcal{N}(0, R)}{\theta_i ~ N(0, R)}
  \deqn{\epsilon_i \sim \mathcal{N}(0, \Sigma)}{\epsilon_i ~ N(0, \Sigma)}
  \deqn{\Sigma = diag(\sigma^2_1, ..., \sigma^2_M)}
where \eqn{Y^*_i}{Y*_i} is the \eqn{M}-vector containing the latent
variables underlying the corresponding \eqn{M} manifest variables
\eqn{Y_i}, which can be continuous such that
\eqn{Y_{im} = Y^*_{im}}{Y_im = Y*_im}, or binary with
\eqn{Y_{im} = 1[Y^*_{im} > 0]}{Y_im = 1[Y*_im > 0]}, for \eqn{m = 1, ..., M}.
The \eqn{K}-vector \eqn{\theta_i} contains the latent factors, and
\eqn{\alpha} is the \eqn{(M \times K)}{(M*K)}-matrix of factor loadings.
The \eqn{M}-vector \eqn{\epsilon_i} is the vector of error terms.
Covariates can be included in the \eqn{Q}-vector \eqn{X_i} and are related to
the manifest variables through the \eqn{(M \times Q)}{(M*Q)}-matrix of
regression coefficients \eqn{\beta}.

The number of latent factors \eqn{K} is specified as \code{Kmax}. However,
during MCMC sampling the stochastic search process on the matrix \eqn{\alpha}
may produce zero columns, thereby reducing the number of active factors.

The covariance matrix \eqn{R} of the latent factors is assumed to be a
correlation matrix for identification.

Each row of the factor loading matrix \eqn{\alpha} contains at most one
nonzero element (dedicated factor model). The allocation of the manifest
variables to the latent factors is indicated by the binary matrix
\eqn{\Delta} with same dimensions as \eqn{\alpha}, such that each row
\eqn{\Delta_m} indicates which factor loading is different from zero, e.g.:
\deqn{\Delta_m = (0, .., 0, 1, 0, ..., 0) \equiv e_k}{
      \Delta_m = (0, .., 0, 1, 0, ..., 0) = e_k}
indicates that variable \eqn{m} loads on the \eqn{k}th factor, where
\eqn{e_k} is a \eqn{K}-vector that contains only zeros, besides its \eqn{k}th
element that equals 1.

\strong{Identification.} The function verifies that the maximum number of
latent factors \code{Kmax} does not exceed the Ledermann bound. It also
checks that \code{Kmax} is consistent with the identification restriction
specified with \code{Nid} (enough variables should be available to load on
the factors when \code{Kmax} is reached). The default value for \code{Kmax}
is the minimum between the Ledermann bound and the maximum number of factors
that can be loaded by \code{Nid} variables. The user is free to select the
level of identification, see CFSHP section 2.2 (non-identified models are
allowed with \code{Nid = 1}).

\strong{Prior specification.}
The indicators are assumed to have the following probabilities,
for \eqn{k = 1, ..., K}:
\deqn{Prob(\Delta_m = e_k \mid \tau_k) = \tau_k}{
      Prob(\Delta_m = e_k | \tau_k) = \tau_k}
\deqn{\tau = (\tau_0, \tau_1, ..., \tau_K)}
where the probabilities, if \code{indp.tau0 = FALSE}, are specified as:
\deqn{\tau = [\tau_0, (1-\tau_0)\tau^*_1, ..., (1-\tau_0)\tau^*_K]}{
      \tau = [\tau_0, (1-\tau_0)\tau*_1, ..., (1-\tau_0)\tau*_K]}
\deqn{\tau_0 \sim \mathcal{B}eta(\kappa_0, \xi_0)}{
      \tau_0 ~ Beta(\kappa_0, \xi_0)}
\deqn{\tau^* = (\tau^*_1, ..., \tau^*_K) \sim \mathcal{D}ir(\kappa)}{
      \tau* = (\tau*_1, ..., \tau*_K) ~ Dir(\kappa)}
with \eqn{\kappa_0} = \code{kappa0}, \eqn{\xi_0} = \code{xi0} and
\eqn{\kappa} = \code{kappa}.
Alternatively, if \code{indp.tau0 = TRUE}, the probabilities are specified
as:
\deqn{\tau_m = [\tau_{0m}, (1-\tau_{0m})\tau^*_1, ...,
        (1-\tau_{0m})\tau^*_K]}{
      \tau_m = [\tau_0m, (1-\tau_0m)\tau*_1, ..., (1-\tau_0m)\tau*_K]}
\deqn{\tau_{0m} \sim \mathcal{B}eta(\kappa_0, \xi_0)}{
      \tau_0m ~ Beta(\kappa_0, \xi_0)}
for each manifest variable \eqn{m = 1, ..., M}.

A normal-inverse-Gamma prior distribution is assumed on the nonzero factor
loadings and on the idiosyncratic variances:
\deqn{\sigma^2_m \sim \mathcal{I}nv-\mathcal{G}amma(c_{0m}, C_{0m})}{
      \sigma^2_m ~ Inv-Gamma(c0_m, C0_m)}
\deqn{\alpha_m^\Delta \mid \sigma^2_m \sim \mathcal{N}(0, A_{0m}\sigma^2_m)}{
      \alpha_m^\Delta | \sigma^2_m ~ N(0, A0_m * \sigma^2_m)}
where \eqn{\alpha_m^\Delta} denotes the only nonzero loading in the \eqn{m}th
row of \eqn{\alpha}.

For the regression coefficients, a multivariate Normal prior distribution is
assumed on each row \eqn{m = 1, ..., M} of \eqn{\beta}:
\deqn{\beta_m \sim \mathcal{N}(0, B_0 I_Q)}{\beta_m ~ N(0, B_0 I_Q)}
The covariates can be different across manifest variables, implying zero
restrictions on the matrix \eqn{\beta}. To specify covariates, use a list
of formulas as \code{model} (see example below).

To sample the correlation matrix \eqn{R} of the latent factors, marginal data
augmentation (van Dyk and Meng, 2001) is implemented, see CFSHP section 2.2.
Using the transformation \eqn{\Omega = \Lambda^{1/2} R \Lambda^{1/2}}, the
parameters \eqn{\Lambda = diag(\lambda_1, ..., \lambda_K)} are used as
\emph{working parameters}. These parameters correspond to the variances of
the latent factors in an expanded version of the model where the factors do
not have unit variances. Two prior distributions can be specified on the
covariance matrix \eqn{\Omega} in the expanded model:
\itemize{
  \item Inverse-Wishart distribution if \code{HW.prior = FALSE}:
  \deqn{\Omega \sim \mathcal{I}nv-\mathcal{W}ishart(\nu_0, diag(S_0))}{
        \Omega ~ Inv-Wishart(\nu_0, diag(S0))}
  with \eqn{\nu_0} = \code{nu0} and \eqn{S_0} = \code{S0}.
  \item Huang-Wand (2013) prior if \code{HW.prior = TRUE}:
  \deqn{\Omega \sim \mathcal{I}nv-\mathcal{W}ishart(\nu_0, W), \qquad
        W = diag(w_1, ..., w_K)}{
        \Omega ~ Inv-Wishart(nu0, W), W = diag(w_1, ..., w_K)}
  \deqn{w_k \sim \mathcal{G}amma\left(\frac{1}{2},
        \frac{1}{2\nu^*S_{0k}}\right)}{w_k ~ Gamma(1/2, 1/(2\nu*S0_k))}
  with \eqn{\nu^*}{\nu*} = \code{nu0} - \code{Kmax} + 1, and the shape and
  rate parameters are specified such that the mean of the gamma distribution
  is equal to \eqn{\nu^* S_{0k}}{\nu* S0_k}, for each \eqn{k = 1, ..., K}.
}

\strong{Missing values.} Missing values (\code{NA}) are allowed in the
manifest variables. They are drawn from their corresponding conditional
distributions during MCMC sampling. Control variables with missing values
can be passed to the function. However, all the observations with at least
one missing value in the covariates are discarded from the sample (a warning
message is issued in that case).
}
\examples{
#### model without covariates

set.seed(6)

# generate fake data with 15 manifest variables and 3 factors
N <- 200    # number of observations
Y <- simul.dedic.facmod(N, dedic = rep(1:3, each = 5))

# run MCMC sampler
# notice: 1000 MCMC iterations for illustration purposes only,
#  increase this number to obtain reliable posterior results!
mcmc <- befa(Y, Kmax = 5, iter = 1000)

# post process MCMC draws to restore identification
mcmc <- post.column.switch(mcmc)
mcmc <- post.sign.switch(mcmc)

# summarize highest posterior probability (HPP) model
hppm <- HPPmodel(mcmc)
hppm$prob                                # HPP model probability
hppm$dedic                               # indicators in HPP model
colMeans(hppm$draws)                     # posterior means in HPP model
attributes(Y)[c('alpha', 'sigma', 'R')]  # true model parameters

#### model with covariates

# generate covariates and regression coefficients
Xcov <- cbind(1, matrix(rnorm(4*N), ncol = 4))
colnames(Xcov) <- c('(Intercept)', paste0('X', 1:4))
beta <- rbind(rnorm(15), rnorm(15), diag(3) \%x\% t(rnorm(5)))

# add covariates to previous model
Y <- Y + Xcov \%*\% beta

# specify model
model <- c('~ X1',                           # X1 covariate in all equations
           paste0('Y', 1:5,   ' ~ X2'),      # X2 covariate for Y1-Y5 only
           paste0('Y', 6:10,  ' ~ X3'),      # X3 covariate for Y6-Y10 only
           paste0('Y', 11:15, ' ~ X4'))      # X4 covariate for Y11-Y15 only
model <- lapply(as.list(model), as.formula)  # make list of formulas

# run MCMC sampler
mcmc <- befa(model, data = data.frame(Y, Xcov), Kmax = 5, iter = 1000)

# compare posterior mean of regression coefficients to true values
post <- cbind(beta[beta != 0], colMeans(mcmc$draws)[31:75])
colnames(post) <- c('true', 'mcmc')
post

}
\author{
Rémi Piatek \email{remi.piatek@econ.ku.dk}
}
\references{
G. Conti, S. Frühwirth-Schnatter, J.J. Heckman, R. Piatek (2014):
``Bayesian Exploratory Factor Analysis'', \emph{Journal of Econometrics},
183(1), pages 31-57, \url{http://dx.doi.org/10.1016/j.jeconom.2014.06.008}.

A. Huang, M.P. Wand (2013):
``Simple Marginally Noninformative Prior Distributions for Covariance
Matrices'', \emph{Bayesian Analysis}, 8(2), pages 439-452,
\url{http://dx.doi.org/10.1214/13-BA815}.

D.A. van Dyk, X.-L. Meng (2001):
``The Art of Data Augmentation'',
\emph{Journal of Computational and Graphical Statistics}, 10(1), pages 1-50,
\url{http://dx.doi.org/10.1198/10618600152418584}.
}
\seealso{
\code{\link{post.column.switch}} and \code{\link{post.sign.switch}}
for column switching and sign switching of the factor loading matrix and of
the correlation matrix of the latent factors to restore identification
\emph{a posteriori}.

\code{\link{HPPmodel}} to find the highest posterior probability
model.
}

