use core::num::Wrapping;
use core::ops::Neg;

use crate::float::FloatCore;
use crate::Num;


pub trait Signed: Sized + Num + Neg<Output = Self> {





    fn abs(&self) -> Self;





    fn abs_sub(&self, other: &Self) -> Self;














    fn signum(&self) -> Self;


    fn is_positive(&self) -> bool;


    fn is_negative(&self) -> bool;
}

macro_rules! signed_impl {
    ($($t:ty)*) => ($(
        impl Signed for $t {
            #[inline]
            fn abs(&self) -> $t {
                if self.is_negative() { -*self } else { *self }
            }

            #[inline]
            fn abs_sub(&self, other: &$t) -> $t {
                if *self <= *other { 0 } else { *self - *other }
            }

            #[inline]
            fn signum(&self) -> $t {
                match *self {
                    n if n > 0 => 1,
                    0 => 0,
                    _ => -1,
                }
            }

            #[inline]
            fn is_positive(&self) -> bool { *self > 0 }

            #[inline]
            fn is_negative(&self) -> bool { *self < 0 }
        }
    )*)
}

signed_impl!(isize i8 i16 i32 i64 i128);

impl<T: Signed> Signed for Wrapping<T>
where
    Wrapping<T>: Num + Neg<Output = Wrapping<T>>,
{
    #[inline]
    fn abs(&self) -> Self {
        Wrapping(self.0.abs())
    }

    #[inline]
    fn abs_sub(&self, other: &Self) -> Self {
        Wrapping(self.0.abs_sub(&other.0))
    }

    #[inline]
    fn signum(&self) -> Self {
        Wrapping(self.0.signum())
    }

    #[inline]
    fn is_positive(&self) -> bool {
        self.0.is_positive()
    }

    #[inline]
    fn is_negative(&self) -> bool {
        self.0.is_negative()
    }
}

macro_rules! signed_float_impl {
    ($t:ty) => {
        impl Signed for $t {

            #[inline]
            fn abs(&self) -> $t {
                FloatCore::abs(*self)
            }




            #[inline]
            fn abs_sub(&self, other: &$t) -> $t {
                if *self <= *other {
                    0.
                } else {
                    *self - *other
                }
            }






            #[inline]
            fn signum(&self) -> $t {
                FloatCore::signum(*self)
            }


            #[inline]
            fn is_positive(&self) -> bool {
                FloatCore::is_sign_positive(*self)
            }


            #[inline]
            fn is_negative(&self) -> bool {
                FloatCore::is_sign_negative(*self)
            }
        }
    };
}

signed_float_impl!(f32);
signed_float_impl!(f64);






#[inline(always)]
pub fn abs<T: Signed>(value: T) -> T {
    value.abs()
}





#[inline(always)]
pub fn abs_sub<T: Signed>(x: T, y: T) -> T {
    x.abs_sub(&y)
}














#[inline(always)]
pub fn signum<T: Signed>(value: T) -> T {
    value.signum()
}


pub trait Unsigned: Num {}

macro_rules! empty_trait_impl {
    ($name:ident for $($t:ty)*) => ($(
        impl $name for $t {}
    )*)
}

empty_trait_impl!(Unsigned for usize u8 u16 u32 u64 u128);

impl<T: Unsigned> Unsigned for Wrapping<T> where Wrapping<T>: Num {}

#[test]
fn unsigned_wrapping_is_unsigned() {
    fn require_unsigned<T: Unsigned>(_: &T) {}
    require_unsigned(&Wrapping(42_u32));
}

#[test]
fn signed_wrapping_is_signed() {
    fn require_signed<T: Signed>(_: &T) {}
    require_signed(&Wrapping(-42));
}
