#' Exploitation Probability
#'
#' Converts capture probabilities into exploitation probabilities based on the
#' release and handling mortality probabilities where
#' the probability of exploitation includes handling mortalities.
#' The calculation assumes that a released fish cannot be recaught in the same year.
#'
#' In the case of no release (or 100% handling mortalities) the exploitation probability
#' is identical to the capture probability.
#' Otherwise it is less.
#'
#' @inheritParams params
#' @param pi A vector of capture probabilities to calculate the exploitation probabilities for.
#' @return A vector of exploitation probabilities.
#' @seealso [ypr_population()]
#' @export
#' @examples
#' ypr_exploitation(ypr_population(pi = 0.4))
#' ypr_exploitation(ypr_population(pi = 0.4, rho = 0.6, Hm = 0.2))
ypr_exploitation <- function(population, pi = population$pi) {
  chk_population(population)
  chk_numeric(pi)
  chk_not_any_na(pi)
  chk_range(pi)
  population$pi <- pi

  u <- with(population, {
    pi * (1 - rho) + pi * rho * Hm
  })
  u
}
