##' DFT spectrum function adapted from libassp
##'
##' Short-term spectral analysis of the signal in <listOfFiles>
##' using the Fast Fourier Transform. The default is to
##' calculate an unsmoothed narrow-band spectrum with the
##' size of the analysis window equal to the length of the
##' FFT. The output from the FFT will be converted to a
##' power spectrum in dB from 0 Hz up to and including the
##' Nyquist rate.
##' Analysis results will be written to a file with the
##' base name of the input file and the spectrum type in
##' lower case as extension (e.g. '.dft').
##' Default output is in SSFF format with the
##' spectrum type in lower case as track name.
##' @title dftSpectrum
##' @param listOfFiles vector of file paths to be processed by function 
##' @param optLogFilePath path to option log file
##' @param beginTime = <time>: set begin of analysis interval to <time> seconds
##' (default: begin of data)
##' @param centerTime = <time>: set single-frame analysis with the analysis
##' window centred at <time> seconds; overrules beginTime, endTime and
##' windowShift options
##' @param endTime = <time>: set end of analysis interval to <time> seconds
##' (default: end of data)
##' @param resolution = <freq>: set FFT length to the smallest value which
##' results in a frequency resolution of <freq> Hz or better (default: 40.0)
##' @param fftLength = <num>: set FFT length to <num> points (overrules default
##' and 'resolution' option)
##' @param windowShift = <dur>: set analysis window shift to <dur> ms
##' (default: 5.0)
##' @param window = <type>: set analysis window function to <type> (default:
##' BLACKMAN)
##' @param bandwidth = <freq>: set the effective analysis bandwidth to <freq>
##' Hz (default: 0, yielding the smallest possible value given the length of
##' the FFT)
##' @param toFile write results to file (default extension depends on )
##' @param explicitExt set if you wish to overwride the default extension
##' @param outputDirectory directory in which output files are stored. Defaults to NULL, i.e.
##' the directory of the input files
##' @param forceToLog is set by the global package variable useWrasspLogger. This is set
##' to FALSE by default and should be set to TRUE is logging is desired.
##' @param verbose display infos & show progress bar
##' @return nrOfProcessedFiles or if only one file to process return
##' AsspDataObj of that file
##' @author Raphael Winkelmann
##' @author Lasse Bombien
##' @seealso \code{\link{cssSpectrum}}, \code{\link{lpsSpectrum}}, \code{\link{cepstrum}}; 
##' all derived from libassp's spectrum function.
##' @useDynLib wrassp
##' @examples
##' # get path to audio file
##' path2wav <- list.files(system.file("extdata", package = "wrassp"), 
##'                        pattern = glob2rx("*.wav"), 
##'                        full.names = TRUE)[1]
##' 
##' # calculate dft spectrum
##' res <- dftSpectrum(path2wav, toFile=FALSE)
##' 
##' # plot spectral values at midpoint of signal
##' plot(res$dft[dim(res$dft)[1]/2,], 
##'      type='l', 
##'      xlab='spectral value index', 
##'      ylab='spectral value')
##'      
##' @export
'dftSpectrum' <- function(listOfFiles = NULL, optLogFilePath = NULL,
                          beginTime = 0.0, centerTime = FALSE,
                          endTime = 0.0, resolution = 40.0,
                          fftLength = 0, windowShift = 5.0, 
                          window = 'BLACKMAN', bandwidth = 0.0, ## DFT specific
                          toFile = TRUE, explicitExt = NULL, 
                          outputDirectory = NULL, forceToLog = useWrasspLogger,
                          verbose = TRUE) {
  ## ########################
  ## a few parameter checks and expand paths
  
  if (is.null(listOfFiles)) {
    stop(paste("listOfFiles is NULL! It has to be a string or vector of file",
               "paths (min length = 1) pointing to valid file(s) to perform",
               "the given analysis function."))
  }
  
  if (is.null(optLogFilePath) && forceToLog){
    stop("optLogFilePath is NULL! -> not logging!")
  }else{
    if(forceToLog){
      optLogFilePath = path.expand(optLogFilePath)  
    }
  }
  
  if(!isAsspWindowType(window)){
    stop("WindowFunction of type '", window,"' is not supported!")
  }
  
  if (!is.null(outputDirectory)) {
    outputDirectory = normalizePath(path.expand(outputDirectory))
    finfo  <- file.info(outputDirectory)
    if (is.na(finfo$isdir))
      if (!dir.create(outputDirectory, recursive=TRUE))
        stop('Unable to create output directory.')
    else if (!finfo$isdir)
      stop(paste(outputDirectory, 'exists but is not a directory.'))
  }
  
  ###########################
  # Pre-process file list
  listOfFiles <- prepareFiles(listOfFiles)
  
  ## #######################
  ## perform analysis
  
  if(length(listOfFiles)==1 | !verbose){
    pb <- NULL
  }else{
    if(toFile==FALSE){
      stop("length(listOfFiles) is > 1 and toFile=FALSE! toFile=FALSE only permitted for single files.")
    }
    cat('\n  INFO: applying dftSpectrum to', length(listOfFiles), 'files\n')
    pb <- utils::txtProgressBar(min = 0, max = length(listOfFiles), style = 3)
  } 
  
  externalRes = invisible(.External("performAssp", listOfFiles, 
                                    fname = "spectrum", beginTime = beginTime, 
                                    centerTime = centerTime, endTime = endTime, 
                                    resolution = resolution, 
                                    fftLength = as.integer(fftLength),
                                    windowShift = windowShift, window = window, 
                                    bandwidth = bandwidth, 
                                    toFile = toFile, explicitExt = explicitExt, 
                                    progressBar = pb, outputDirectory = outputDirectory,
                                    PACKAGE = "wrassp"))
  
  
  ## #########################
  ## write options to options log file
  if (forceToLog){
    optionsGivenAsArgs = as.list(match.call(expand.dots = TRUE))
    wrassp.logger(optionsGivenAsArgs[[1]], optionsGivenAsArgs[-1],
                  optLogFilePath, listOfFiles)
    
  }
  
  ## #########################
  ## return dataObj if length only one file
  
  if(!is.null(pb)){
    close(pb)
  }else{
    return(externalRes)
  }
  
}
