#' Visual Studio Project Get Work Items
#'
#' @description
#' These functions will allow you to scrape work item information from a particular Visual Studio project.
#'
#' @details
#' For more information about work item API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/wit/work\%20items}.
#'
#' @param domain the location of the visual studio server
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param id (for single work item request) ID of the work item to retrieve
#' @param query a list of extra parameters that can be sent to the API call:
#' \describe{
#' \item{\code{ids}}{[character] a comma-separated list of up to 200 IDs of the work items to get}
#' \item{\code{fields}}{[character] a comma-separated list of up to 100 fields to get with each work item.
#' If not specified, all fields with values are returned. Calculated fields such as Attached File Count must be specifically
#' queried for using this parameter.}
#' \item{\code{asOf}}{[Date] gets the work items as they existed at this time}
#' \item{\code{ErrorPolicy}}{[character] determines if the call will throw an error when encountering a work item (default behavior)
#' that doesn't exist (\code{throw}) or simply omit it (\code{omit})}
#' }
#'
#' @rdname vsts_get_wk
#' @export
vsts_get_workitems <- function(domain, auth_key, query = NULL) {
  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/_apis/wit/workitems?api-version=1.0')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key), query = query)
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', 'get work items list')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  return(invisible(content))
}

#' @rdname vsts_get_wk
#' @export
vsts_get_workitem <- function(domain, auth_key, id) {
  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/_apis/wit/workitems/', id, '?api-version=1.0')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', paste0('get work item #', id))$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  return(invisible(content))
}

#' Visual Studio Project Work Items
#'
#' @description
#' These functions will allow you to scrape work item information from a particular Visual Studio project.
#'
#' @details
#' For more information about work item API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/wit/work\%20items}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param item_type the type of work item to be created
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param ... arguments passed to \link{vsts_get_workitem_fields}
#'
#' @rdname vsts_create_wk
#' @export
vsts_create_workitem <- function(domain, project, item_type, auth_key, ...) {
  #Get work item types to reference
  item_types <- vsts_get_itemtypes(domain, project, auth_key) %>% .$name
  item_type <- item_types[match(tolower(item_type), tolower(item_types))]
  if(is.na(item_type)) {
    cat('item_type not available for project. Select from:', paste(item_types, collapse = ', '), '\n')
    return(invisible(NULL))
  }

  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/', project, '/_apis/wit/workitems/$',
                gsub(' ', '%20', item_type), '?api-version=1.0')

  content_info <- vsts_get_workitem_fields(...)
  content_body <- jsonlite::toJSON(apply(content_info, 1, function(x)
    list(op = 'add', path = paste0('/fields/', x['Field']), value = x['Value'])), auto_unbox = TRUE)

  response <- httr::PATCH(uri, httr::add_headers(Authorization = auth_key),
                          httr::content_type('application/json-patch+json'), body = content_body)
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', paste('add', item_type, 'to', project))$message, '\n')
    return(invisible(NULL))
  }

  content <- as.data.frame(httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE))
  return(invisible(content))
}

vsts_get_itemtypes <- function(domain, project, auth_key) {
  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/', project, '/_apis/wit/workItemTypes?api-version=1.0')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', 'get item types')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  return(invisible(content))
}

#' Visual Studio Work Item Fields
#'
#' @description
#' This contains all the fields required of any work item in a visual studio project and helps add/rename the fields of the selected
#' work item.
#'
#' @details
#' For more information about work item fields API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/wit/fields}.
#'
#' @param System.Title [character] title of the Visual Studio work item
#' @param System.Description [character] description of the Visual Studio work item
#' @param System.TeamProject [character] name of the Visual Studio project
#' @param System.AreaPath [character] path of the Visual Studio work item
#' @param System.IterationPath [character] name of the Visual Studio iteration path
#' @param Microsoft.VSTS.Common.Priority [integer] priority of the work item - 1 to 4
#' @param ... other fields that might have been missed out originally
#'
#' @export
vsts_get_workitem_fields <- function(System.Title, System.Description, System.TeamProject,
                                     System.AreaPath, System.IterationPath,
                                     Microsoft.VSTS.Common.Priority, ...) {
  field_list <- as.list(match.call())
  field_list <- field_list[-1]
  data.frame(Field = names(field_list), Value = unlist(field_list, use.names = FALSE))
}
