% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eocusum_sim.R
\name{eocusum_crit_sim}
\alias{eocusum_crit_sim}
\title{Compute alarm threshold of EO-CUSUM control charts using simulation}
\usage{
eocusum_crit_sim(L0, k, df, coeff, m = 100, yemp = TRUE,
  side = "low", nc = 1, jmax = 4, verbose = FALSE)
}
\arguments{
\item{L0}{Double. Prespecified in-control Average Run Length.}

\item{k}{Double. Reference value of the CUSUM control chart. Either \code{0} or a positive
value. Can be determined with function \code{\link{optimal_k}}.}

\item{df}{Data Frame. First column are Parsonnet Score values within a range of \code{0} to
\code{100} representing the preoperative patient risk. The second column are binary (0/1)
outcome values of each operation.}

\item{coeff}{Numeric Vector. Estimated coefficients \eqn{\alpha}{alpha} and \eqn{\beta}{beta}
from the binary logistic regression model. For more information see details.}

\item{m}{Integer. Number of simulation runs.}

\item{yemp}{Logical. If \code{TRUE} use observed outcome value, if \code{FALSE} use estimated
binary logistc regression model.}

\item{side}{Character. Default is \code{"low"} to calculate ARL for the upper arm of the V-mask.
If side = \code{"up"}, calculate the lower arm of the V-mask.}

\item{nc}{Integer. Number of cores.}

\item{jmax}{Integer. Number of digits for grid search.}

\item{verbose}{Logical. If \code{TRUE} verbose output is included, if \code{FALSE} a quiet
calculation of \code{h} is done.}
}
\value{
Returns a single value which is the control limit \code{h} for a given ARL.
}
\description{
Compute alarm threshold of EO-CUSUM control charts using simulation.
}
\details{
The function \code{eocusum_crit_sim} determines the control limit for given in-control
 ARL (\code{L0}) by applying a multi-stage search procedure which includes secant rule and the
  parallel version of \code{\link{eocusum_arl_sim}} using \code{\link{mclapply}}.
}
\examples{
\dontrun{
data("cardiacsurgery", package = "spcadjust")
library("dplyr")

## preprocess data to 30 day mortality and subset phase I (In-control) of surgeons 2
S2I <- cardiacsurgery \%>\% rename(s = Parsonnet) \%>\%
  mutate(y = ifelse(status == 1 & time <= 30, 1, 0),
         phase = factor(ifelse(date < 2*365, "I", "II"))) \%>\%
  filter(phase == "I", surgeon == 2) \%>\% select(s, y)

## estimate coefficients from logit model
coeff1 <- coef(glm(y ~ s, data = S2I, family = "binomial"))
## Number of simulation runs
m <- 10^3
set.seed(1234)
## Number of cores
nc <- parallel::detectCores()

## determine k for detecting deterioration
kopt <- optimal_k(QA = 2, df = S2I, coeff = coeff, yemp = FALSE)

## compute threshold for prespecified in-control ARL
h <- eocusum_crit_sim(L0 = 370, df = S2I, k = kopt, m = m, coeff = coeff1, side = "low",
nc = nc)

## parameters to set up a tabular CUSUM or V-Mask
d <- h/kopt
theta <- atan(kopt)*180/pi
cbind(kopt, h, theta, d)
}
}
\references{
Barnard GA (1959). Control charts and stochastic processes.
\emph{J R Stat Soc Series B Stat Methodol}, \strong{21}(2), pp. 239--271.

Kemp KW (1961). The Average Run Length of the Cumulative Sum Chart
when a V-mask is used. \emph{J R Stat Soc Series B Stat Methodol}, \strong{23}(1),pp. 149--153.

Wittenberg P, Gan FF, Knoth S (2018).
A simple signaling rule for variable life-adjusted display derived from
an equivalent risk-adjusted CUSUM chart.
\emph{Statistics in Medicine}, \strong{37}(16), pp 2455--2473.
}
\author{
Philipp Wittenberg
}
