% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/usl.R
\name{usl}
\alias{usl}
\title{Create a model for the Universal Scalability Law}
\usage{
usl(formula, data, method = "default")
}
\arguments{
\item{formula}{An object of class "\code{\link{formula}}" (or one that
can be coerced to that class): a symbolic description of the model to be
analyzed. The details of model specification are given under 'Details'.}

\item{data}{A data frame, list or environment (or object coercible by
as.data.frame to a data frame) containing the variables in the model.
If not found in data, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{usl} is called.}

\item{method}{Character value specifying the method to use. The possible
values are described under 'Details'.}
}
\value{
An object of class USL.
}
\description{
\code{usl} is used to create a model for the Universal Scalability Law.
}
\details{
The Universal Scalability Law is used to forcast the scalability of
either a hardware or a software system.

The USL model works with one independent variable (e.g. virtual users,
processes, threads, ...) and one dependent variable (e.g. throughput, ...).
Therefore the model formula must be in the simple
"\code{response ~ predictor}" format.

The model produces two main coefficients as result: \code{alpha} models the
contention and \code{beta} the coherency delay of the system. The third
coefficient \code{gamma} estimates the value of the dependent variable
(e.g. throughput) for the single user/process/thread case. It therefore
corresponds to the scale factor calculated in previous versions of the
\code{usl} package.

The function \code{\link{coef}} extracts the coefficients from the model
object.

The argument \code{method} selects which solver is used to solve the
model:

\itemize{
  \item "\code{nls}" for a nonlinear regression model. This method
    estimates all coefficients \code{alpha}, \code{beta} and \code{gamma}.
    The R base function \code{\link{nls}} with the "\code{port}" algorithm
    is used internally to solve the model. So all restrictions of the
    "\code{port}" algorithm apply.
  \item "\code{nlxb}" for a nonliner regression model using the function
    \code{\link{nlxb}} from the \code{\link{nlsr}} package. This method
    also estimates all three coefficients. It is expected to be more robust
    than the \code{nls} method.
  \item "\code{default}" for the default method using a transformation
    into a 2nd degree polynom has been removed with the implementation
    of the model using three coefficients in the \pkg{usl} package 2.0.0.
    Calling the "\code{default}" method will internally dispatch to the
    "\code{nlxb}" solver instead.
}

The Universal Scalability Law can be expressed with following formula.
\code{C(N)} predicts the relative capacity of the system for a given
load \code{N}:

\deqn{C(N) = \frac{\gamma N}{1 + \alpha (N - 1) + \beta N (N - 1)}}{C(N) = (\gamma N) / (1 + \alpha * (N - 1) + \beta * N * (N - 1))}
}
\examples{
require(usl)

data(raytracer)

## Create USL model for "throughput" by "processors"
usl.model <- usl(throughput ~ processors, raytracer)

## Show summary of model parameters
summary(usl.model)

## Show complete list of efficiency parameters
efficiency(usl.model)

## Extract coefficients for model
coef(usl.model)

## Calculate point of peak scalability
peak.scalability(usl.model)

## Plot original data and scalability function
plot(raytracer)
plot(usl.model, add=TRUE)

}
\references{
Neil J. Gunther. Guerrilla Capacity Planning: A Tactical
  Approach to Planning for Highly Scalable Applications and Services.
  Springer, Heidelberg, Germany, 1st edition, 2007.

John C. Nash. nlsr: Functions for nonlinear least squares
  solutions, 2017. R package version 2017.6.18.
}
\seealso{
\code{\link{efficiency,USL-method}},
  \code{\link{scalability,USL-method}},
  \code{\link{peak.scalability,USL-method}},
  \code{\link{optimal.scalability,USL-method}},
  \code{\link{limit.scalability,USL-method}},
  \code{\link{summary,USL-method}},
  \code{\link{sigma,USL-method}}
  \code{\link{predict,USL-method}},
  \code{\link{overhead,USL-method}},
  \code{\link{confint,USL-method}},
  \code{\link{coef}},
  \code{\link{fitted}},
  \code{\link{residuals}},
  \code{\link{df.residual}}
}
