% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ashdog.R
\name{flexdog}
\alias{flexdog}
\title{Flexible genotyping for polyploids from next-generation sequencing data.}
\usage{
flexdog(refvec, sizevec, ploidy, model = c("norm", "flex", "hw", "bb", "ash",
  "f1", "s1", "f1pp", "s1pp", "uniform"), p1ref = NULL, p1size = NULL,
  p2ref = NULL, p2size = NULL, bias_init = exp(c(-1, -0.5, 0, 0.5, 1)),
  verbose = TRUE, outliers = FALSE, ...)
}
\arguments{
\item{refvec}{A vector of counts of reads of the reference allele.}

\item{sizevec}{A vector of total counts.}

\item{ploidy}{The ploidy of the species. Assumed to be the same for each
individual.}

\item{model}{What form should the prior (genotype distribution) take?
See Details for possible values.}

\item{p1ref}{The reference counts for the first parent if
\code{model = "f1"} (or \code{model = "f1pp"}), or for
the only parent if \code{model = "s1"} (or
\code{model = "s1pp"}).}

\item{p1size}{The total counts for the first parent if
\code{model = "f1"} (or \code{model = "f1pp"}),
or for the only parent if \code{model = "s1"}
(or \code{model = "s1pp"}).}

\item{p2ref}{The reference counts for the second parent if
\code{model = "f1"} (or \code{model = "f1pp"}).}

\item{p2size}{The total counts for the second parent if
\code{model = "f1"} (or \code{model = "f1pp"}).}

\item{bias_init}{A vector of initial values for the bias parameter
over the multiple runs of \code{flexdog_full}.}

\item{verbose}{Should we output more (\code{TRUE}) or less
(\code{FALSE})?}

\item{outliers}{A logical. Should we allow for the inclusion of outliers
(\code{TRUE}) or not (\code{FALSE}). Only supported when
\code{model = "f1"} or \code{model = "s1"}. I wouldn't
recommend it for any other model anyway.}

\item{...}{Additional parameters to pass to \code{\link{flexdog_full}}.}
}
\value{
An object of class \code{flexdog}, which consists
    of a list with some or all of the following elements:
\describe{
  \item{\code{bias}}{The estimated bias parameter.}
  \item{\code{seq}}{The estimated sequencing error rate.}
  \item{\code{od}}{The estimated overdispersion parameter.}
  \item{\code{num_iter}}{The number of EM iterations ran. You should
      be wary if this equals \code{itermax}.}
  \item{\code{llike}}{The maximum marginal log-likelihood.}
  \item{\code{postmat}}{A matrix of posterior probabilities of each
      genotype for each individual. The rows index the individuals
      and the columns index the allele dosage.}
  \item{\code{gene_dist}}{The estimated genotype distribution. The
      \code{i}th element is the proportion of individuals with
      genotype \code{i-1}. If \code{outliers = TRUE}, then this
      is conditional on the point not being an outlier.}
  \item{\code{par}}{A list of the final estimates of the parameters
      of the genotype distribution. If \code{model = "hw"} then
      this will consist of \code{alpha}, the allele frequency.
      If \code{model = "f1"} or \code{model = "s1"} then this will
      consist of the parent genotype(s), the value of \code{fs1_alpha}
      (now just called \code{alpha}), and possibly the outlier
      proportion \code{out_prop}. If
      \code{model = "bb"} then this will consist of \code{alpha},
      the allele frequency, and \code{tau}, the overdispersion parameter.
      If \code{model = "norm"} then this will consist of \code{mu}, the
      normal mean, and \code{sigma}, the normal standard deviation (not variance).}
  \item{\code{geno}}{The posterior mode genotype. These are your
      genotype estimates.}
  \item{\code{maxpostprob}}{The maximum posterior probability. This
      is equivalent to the posterior probability of correctly
      genotyping each individual.}
  \item{\code{postmean}}{The posterior mean genotype. In downstream
      association studies, you might want to consider using these
      estimates.}
  \item{\code{input$refvec}}{The value of \code{refvec} provided by
      the user.}
  \item{\code{input$sizevec}}{The value of \code{sizevec} provided
      by the user.}
  \item{\code{input$ploidy}}{The value of \code{ploidy} provided
      by the user.}
  \item{\code{input$model}}{The value of \code{model} provided by
      the user.}
  \item{\code{input$p1ref}}{The value of \code{p1ref} provided by the user.}
  \item{\code{input$p1size}}{The value of \code{p1size} provided by the user.}
  \item{\code{input$p2ref}}{The value of \code{p2ref} provided by the user.}
  \item{\code{input$p2size}}{The value of \code{p2size} provided by the user.}
  \item{\code{prop_mis}}{The posterior proportion of individuals
      genotyped incorrectly.}
  \item{\code{out_prop}}{The estimated proportion of points that
      are outliers. Only available if \code{outliers = TRUE}.}
  \item{\code{prob_out}}{The ith element is the posterior probability
      that individual i is an outlier. Only available if
      \code{outliers = TRUE}.}
}
}
\description{
Genotype polyploid individuals from next generation
sequencing (NGS) data while assuming the genotype distribution is one of
several forms. \code{flexdog} does this while accounting for allele bias,
overdispersion, sequencing error, and possibly outlying observations
(if \code{model = "f1"} or \code{model = "s1"}).
}
\details{
Possible values of the genotype distribution (values of \code{model}) are:
\describe{
  \item{\code{"hw"}}{A binomial distribution that results from assuming that
      the population is in Hardy-Weinberg equilibrium (HWE). This actually does
      pretty well even when there are minor to moderate deviations from HWE.}
  \item{\code{"bb"}}{A beta-binomial distribution. This is a overdispersed
      version of "hw" and can be derived from a special case of the Balding-Nichols model.}
  \item{\code{"norm"}}{A distribution whose genotype frequencies are proportional
      to the density value of a normal with some mean and some standard deviation.
      Unlike the "bb" option, this will allow for distributions less dispersed than a binomial.
      This seems to be the most robust to violations in modeling assumptions, and so is the
      default.}
  \item{\code{"ash"}}{Any unimodal prior. This will run \code{ploidy} EM algorithms
      with a different center during each optimization. It returns the center (and its fit)
      with the highest likelihood.}
  \item{\code{"f1"}}{This prior assumes the individuals are all
      full-siblings resulting
      from one generation of a bi-parental cross.
      This model assumes
      a particular type of meiotic behavior: polysomic
      inheritance with
      bivalent, non-preferential pairing. Since this is
      a pretty strong
      and well-founded prior, we allow \code{outliers = TRUE}
      when \code{model = "f1"}.}
  \item{\code{"s1"}}{This prior assumes the individuals are
      all full-siblings resulting
      from one generation of selfing. I.e. there is only
      one parent. This model assumes
      a particular type of meiotic behavior: polysomic
      inheritance with
      bivalent, non-preferential pairing.
      Since this is a pretty strong and well-founded prior,
      we allow \code{outliers = TRUE} when \code{model = "s1"}.}
  \item{\code{"f1pp"}}{The same as \code{"f1"} but accounts for possible
      (and arbitrary levels of) preferential
      pairing during meiosis.}
  \item{\code{"s1pp"}}{The same as \code{"s1"} but accounts for
      possible preferential (and arbitrary levels of)
      pairing during meiosis.}
  \item{\code{"flex"}}{Generically any categorical distribution. Theoretically,
      this works well if you have a lot of individuals. In practice, it seems to
      be less robust to violations in modeling assumptions.}
  \item{\code{"uniform"}}{A discrete uniform distribution. This should never
      be used in practice. Please don't email me that \code{flexdog} doesn't
      work if you use this option.}
}

You might think a good default is \code{model = "uniform"} because it is
somehow an "uninformative prior." But it is very informative and tends to
work horribly in practice. The intuition is that it will estimate
the allele bias and sequencing error rates so that the estimated genotypes
are approximately uniform (since we are assuming that they are approximately
uniform). This will usually result in unintuitive genotyping since most
populations don't have a uniform genotype distribution.
I include it as an option only for completeness. Please, please don't use it.

The value of \code{prop_mis} is a very intuitive measure for
the quality of the SNP. \code{prop_mis} is the posterior
proportion of individuals mis-genotyped. So if you want only SNPS
that accurately genotype, say, 95\% of the individuals, you could
discard all SNPs with a \code{prop_mis} under \code{0.95}.

The value of \code{maxpostprob} is a very intuitive measure
for the quality of the genotype estimate of an individual.
This is the posterior probability of correctly genotyping
the individual when using \code{geno} (the posterior mode)
as the genotype estimate. So if you want to correctly genotype,
say, 95\% of individuals, you could discard all individuals
with a \code{maxpostprob} of under \code{0.95}.

In most datasets I've examined, allelic bias is a major issue. However,
you may fit the model assuming no allelic bias by setting
\code{update_bias = FALSE} and \code{bias_init = 1}.

Prior to using \code{flexdog}, during the read-mapping step,
you could try to get rid of allelic bias by
using WASP (\url{https://doi.org/10.1101/011221}). If you are successful
in removing the allelic bias (because its only source was the read-mapping
step), then setting \code{update_bias = FALSE} and \code{bias_init = 1}
would be reasonable. You can visually inspect SNPs for bias by
using \code{\link{plot_geno}}.

\code{flexdog}, like most methods, is invariant to which allele you
label as the "reference" and which you label as the "alternative".
That is, if you set \code{refvec} with the number of alternative
read-counts, then the resulting genotype estimates
will be the estimated allele dosage of the alternative allele.
}
\examples{
\donttest{

## An S1 population where the first individual
## is the parent. Fit assuming outliers.
data("snpdat")
ploidy  <- 6
refvec  <- snpdat$counts[snpdat$snp == "SNP3"]
sizevec <- snpdat$size[snpdat$snp == "SNP3"]
fout    <- flexdog(refvec   = refvec[-1],
                   sizevec  = sizevec[-1],
                   ploidy   = ploidy,
                   model    = "s1",
                   p1ref    = refvec[1],
                   p1size   = sizevec[1],
                   outliers = TRUE)
plot(fout)

}

## A natural population. We will assume a
## normal prior since there are so few
## individuals.
data("uitdewilligen")
ploidy  <- 4
refvec  <- uitdewilligen$refmat[, 1]
sizevec <- uitdewilligen$sizemat[, 1]
fout    <- flexdog(refvec  = refvec,
                   sizevec = sizevec,
                   ploidy  = ploidy,
                   model   = "norm")
plot(fout)



}
\references{
Gerard, David, Luis Felipe Ventorim Ferrao,
Antonio Augusto Franco Garcia, and Matthew Stephens. 2018.
"Harnessing Empirical Bayes and Mendelian Segregation
for Genotyping Autopolyploids from Messy Sequencing Data."
\emph{bioRxiv}. Cold Spring Harbor Laboratory. doi:10.1101/281550.
}
\seealso{
Run \code{browseVignettes(package = "updog")} in R for example usage.
    Other useful functions include:
\describe{
    \item{\code{\link{flexdog_full}}}{For additional parameter options
          when running \code{flexdog}.}
    \item{\code{\link{rgeno}}}{For simulating genotypes under the allowable
          prior models in \code{flexdog}.}
    \item{\code{\link{rflexdog}}}{For simulating read-counts under the
          assumed likelihood model in \code{flexdog}.}
    \item{\code{\link{plot.flexdog}}}{For plotting the output of
          \code{flexdog}.}
    \item{\code{\link{oracle_mis}}}{For calculating the oracle genotyping
          error rates. This is useful for read-depth calculations
          \emph{before} collecting data. After you have data, using
          the value of \code{prop_mis} is better.}
    \item{\code{\link{oracle_cor}}}{For calculating the correlation
          between the true genotypes and an oracle estimator
          (useful for read-depth calculations \emph{before}
          collecting data).}
}
}
\author{
David Gerard
}
