% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.R
\name{umxRAM}
\alias{umxRAM}
\title{Easier path-based SEM modeling.}
\usage{
umxRAM(model = NA, ..., data = NULL, name = NA, comparison = TRUE,
  suffix = "", showEstimates = c("none", "raw", "std", "both",
  "list of column names"), type = c("Auto", "FIML", "cov", "cor", "WLS",
  "DWLS", "ULS"), allContinuousMethod = c("cumulants", "marginals"),
  autoRun = getOption("umx_auto_run"), tryHard = c("no", "yes",
  "mxTryHard", "mxTryHardOrdinal", "mxTryHardWideSearch"),
  refModels = NULL, remove_unused_manifests = TRUE, independent = NA,
  setValues = TRUE, optimizer = NULL, verbose = FALSE)
}
\arguments{
\item{model}{A model to update (or set to string to use as name for new model)}

\item{...}{mx or umxPaths, mxThreshold objects, etc.}

\item{data}{data for the model. Can be an \code{\link{mxData}} or a data.frame}

\item{name}{A friendly name for the model}

\item{comparison}{Compare the new model to the old (if updating an existing model: default = TRUE)}

\item{suffix}{String to append to each label (useful if model will be used in a multi-group model)}

\item{showEstimates}{Whether to show estimates. Defaults to no (alternatives = "raw", "std", etc.)}

\item{type}{One of "Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"}

\item{allContinuousMethod}{"cumulants" or "marginals". Used in all-continuous WLS data to determine if a means model needed.}

\item{autoRun}{Whether to run the model, and return that (default), or just to create it and return without running.}

\item{tryHard}{Default ('no') uses normal mxRun. "yes" uses mxTryHard. Other options: "mxTryHardOrdinal", "mxTryHardWideSearch"}

\item{refModels}{pass in reference models if available. Use FALSE to suppress computing these if not provided.}

\item{remove_unused_manifests}{Whether to remove variables in the data to which no path makes reference (defaults to TRUE)}

\item{independent}{Whether the model is independent (default = NA)}

\item{setValues}{Whether to generate likely good start values (Defaults to TRUE)}

\item{optimizer}{optionally set the optimizer (default NULL does nothing)}

\item{verbose}{Whether to tell the user what latents and manifests were created etc. (Default = FALSE)}
}
\value{
\itemize{
\item \code{\link{mxModel}}
}
}
\description{
\code{umxRAM} expedites creation of path-based models, still without doing invisible things to the model.

Here's an example that makes a model of miles per gallon (mpg) as a function of weight (wt) and engine displacement (disp)
using \code{mtcars} data.\preformatted{m1 = umxRAM("tim", data = mtcars,
	umxPath(c("wt", "disp"), to = "mpg"),
	umxPath("wt", with = "disp"),
	umxPath(v.m. = c("wt", "disp", "mpg"))
)
}

As you can see, most of the work is done by \code{\link[=umxPath]{umxPath()}}. \code{umxRAM} just wraps these paths up, takes the \code{data =} input, and
then internally sets up all the labels and start values for the model, runs it, displays a summary, and a plot!

Try it, or one of the several models in the examples at the bottom of this page.

A common error is to include data in the main list, a bit like
saying \code{lm(y ~ x + df)} instead of \code{lm(y ~ x, data = dd)}.

\strong{nb}: Because it uses the presence of a variable in the data to detect if a variable is latent or not, umxRAM needs data at build time.

\emph{note}: If you are at the "sketching" stage of theory consideration, \code{umxRAM} supports
a simple vector of manifest names to work with.\preformatted{m1 = umxRAM("sketch", data = c("A", "B", "C"),
	umxPath("A", to = "B"),
	umxPath("B", with = "C"),
	umxPath(v.m. = c("A", "B", "C"))
)
}

Will create this figure:

\if{html}{\figure{sketch.png}{options: width="50\%" alt="Figure: sketch.png"}}
\if{latex}{\figure{sketch.pdf}{options: width=7cm}}
}
\details{
\strong{Comparison with mxModel}

umxRAM differs from mxModel in the following ways:
\enumerate{
\item You don't need to set type = "RAM".
\item You don't need to list manifestVars (they are detected from path usage).
\item You don't need to list latentVars (detected as anything in paths but not in \code{mxData}).
\item You add data with \strong{data = } (as elsewhere in R, e.g. \code{\link{lm}}).
\item You don't need to add labels: paths are automatically labelled "a_to_b" etc.
\item You don't need to set start values, they will be done for you.
\item You don't need to mxRun the model: it will run automatically, and print a summary.
\item You don't need to run summary: with \code{autoRun}, it will print a summary.
\item You get a plot of the model.
\item \code{\link{umxPath}} offers powerful verbs to describe paths.
}

\strong{Comparison with other software}

\strong{Start values}. Currently, manifest variable means are set to the observed means, residual variances are set to 80%
of the observed variance of each variable,
and single-headed paths are set to a positive starting value (currently .9).
\emph{note}: The start-value strategy is subject to improvement, and will be documented in the help for umxRAM.

\strong{Black-box software, defaults, and automatic addition of paths}.
Some SEM software does a lot of behind-the-scenes defaulting and path addition. I've explored
similar features (like auto-creating error and exogenous variances using \code{endog.variances = TRUE}
and \code{exog.variances = TRUE}). Also identification helpers like \code{fix = "latents"}
and \code{fix = "firstLoadings"}.

To be honest, these are not only more trouble than they are worth, they encourage errors and
poor modeling. Learning the handful of \code{\link{umxPath}} short cuts allows modeling to
stay both efficient and unambiguous!
}
\examples{

# ============================================
# = 1. Here's a simple example with raw data =
# ============================================
m1 = umxRAM("tim", data = mtcars,
	umxPath(c("wt", "disp"), to = "mpg"),
	umxPath("wt", with = "disp"),
	umxPath(v.m. = c("wt", "disp", "mpg"))
)

# 2. Use parameters to see the parameter estimates and labels
parameters(m1)

# And umxSummary to get standardized parameters, CIs etc from the run model.
umxSummary(m1, show = "std")
# |name           | Std.Estimate| Std.SE|CI                   |
# |:--------------|------------:|------:|:--------------------|
# |wt_to_mpg      |        -0.54|   0.17|-0.54 [-0.89, -0.2]  |
# |disp_to_mpg    |        -0.36|   0.18|-0.36 [-0.71, -0.02] |
# |mpg_with_mpg   |         0.22|   0.07|0.22 [0.08, 0.35]    |
# |wt_with_wt     |         1.00|   0.00|1 [1, 1]             |
# |b1             |         0.89|   0.04|0.89 [0.81, 0.96]    |
# |disp_with_disp |         1.00|   0.00|1 [1, 1]             |

# 3. #' # Of course you can plot the model
plot(m1)
plot(m1, std=TRUE, means=FALSE)
plot(m1, std = TRUE, strip= FALSE, resid = "line")

# ====================================
# = A cov model, with steps laid out =
# ====================================

# *note*: The variance of displacement is in cubic inches and is very large.
# to help the optimizer, one might, say, multiply disp *.016 to work in litres
tmp = mtcars; tmp$disp= tmp$disp *.016

# We can just give the raw data and ask for it to be made into type cov:
m1 = umxRAM("tim", data = tmp, type="cov",
	umxPath(c("wt", "disp"), to = "mpg"),
	umxPath("wt", with = "disp"),
	umxPath(var = c("mpg", "wt", "disp"))
)

# (see ?umxPath for more nifty options making paths...)

# =========================================
# = umxRAM can also accept mxData as data =
# =========================================
# For convenience, list up the manifests you will be using

selVars = c("mpg", "wt", "disp")
myCov = mxData(cov(tmp[, selVars]), type = "cov", numObs = nrow(mtcars) )

m1 = umxRAM("tim", data = myCov,
	umxPath(c("wt", "disp"), to = "mpg"),
	umxPath("wt", with = "disp"),
	umxPath(var = selVars)
)


# =======================
# = umxRAM supports WLS =
# =======================

# 1. Run an all-continuous WLS model
 mw = umxRAM("raw", data = mtcars[, c("mpg", "wt", "disp")], 
	type = "WLS", allContinuousMethod = "cumulants",
 	umxPath(var = c("wt", "disp", "mpg")),
 	umxPath(c("wt", "disp"), to = "mpg"),
 	umxPath("wt", with = "disp"),
     umxPath(var = c("wt", "disp", "mpg"))
 )
# 2. Switch to marginals to support means
 mw = umxRAM("raw", data = mtcars[, c("mpg", "wt", "disp")], 
	type = "WLS", allContinuousMethod= "marginals",
 	umxPath(var = c("wt", "disp", "mpg")),
 	umxPath(c("wt", "disp"), to = "mpg"),
 	umxPath("wt", with = "disp"),
     umxPath(var = c("wt", "disp", "mpg"))
 )

# ===============================
# = Using umxRAM in Sketch mode =
# ===============================
# No data needed: just list variable names!
# Resulting model will be plotted automatically
m1 = umxRAM("what does unique pairs do, I wonder", data = c("A", "B", "C"),
   umxPath(unique.pairs = c("A", "B", "C"))
)

m1 = umxRAM("ring around the rosey", data = c("B", "C"),
  umxPath(fromEach = c("A", "B", "C"))
)

m1 = umxRAM("fromEach with to", data = c("B", "C"),
   umxPath(fromEach = c("B", "C"), to= "D")
)

m1 = umxRAM("CFA_sketch", data = paste0("x", 1:4),
	umxPath("g", to = paste0("x", 1:4)),
	umxPath(var = paste0("x", 1:4)),
	umxPath(v1m0 = "g")
)


# =================================================
# = This is an example of using your own labels:  =
#   umxRAM will not over-ride them                =
# =================================================
m1 = umxRAM("tim", data = myCov,
	umxPath(c("wt", "disp"), to = "mpg"),
	umxPath(cov = c("wt", "disp"), labels = "b1"),
	umxPath(var = c("wt", "disp", "mpg"))
)
omxCheckEquals(m1$S$labels["disp", "wt"], "b1") # label preserved
m1$S$labels
#      mpg             wt            disp
# mpg  "mpg_with_mpg"  "mpg_with_wt" "disp_with_mpg"
# wt   "mpg_with_wt"   "wt_with_wt"  "b1"
# disp "disp_with_mpg" "b1"          "disp_with_disp"
parameters(m1)
}
\references{
\itemize{
\item \url{https://tbates.github.io}, \url{https://github.com/tbates/umx}
}
}
\seealso{
\code{\link{umxPath}}, \code{\link{umxSummary}}, \code{\link{plot}}, \code{\link{parameters}}, \code{\link{umxSuperModel}}

Other Core Modeling Functions: \code{\link{plot.MxLISRELModel}},
  \code{\link{plot.MxModel}}, \code{\link{umxAlgebra}},
  \code{\link{umxMatrix}}, \code{\link{umxModify}},
  \code{\link{umxPath}}, \code{\link{umxRun}},
  \code{\link{umxSuperModel}}, \code{\link{umx}}
}
\concept{Core Modeling Functions}
