% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_summarise.R
\name{time_summarise}
\alias{time_summarise}
\title{A time based extension to \code{dplyr::summarise()}}
\usage{
time_summarise(
  data,
  time = NULL,
  ...,
  time_by = NULL,
  from = NULL,
  to = NULL,
  time_type = getOption("timeplyr.time_type", "auto"),
  include_interval = FALSE,
  .by = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1),
  roll_month = getOption("timeplyr.roll_month", "preday"),
  roll_dst = getOption("timeplyr.roll_dst", "boundary"),
  sort = TRUE
)
}
\arguments{
\item{data}{A data frame.}

\item{time}{Time variable.}

\item{...}{Additional variables to include.}

\item{time_by}{Time unit. \cr
Must be one of the three:
\itemize{
\item string, specifying either the unit or the number and unit, e.g
\code{time_by = "days"} or \code{time_by = "2 weeks"}
\item named list of length one, the unit being the name, and
the number the value of the list, e.g. \code{list("days" = 7)}.
For the vectorized time functions, you can supply multiple values,
e.g. \code{list("days" = 1:10)}.
\item Numeric vector. If time_by is a numeric vector and x is not a date/datetime,
then arithmetic is used, e.g \code{time_by = 1}.
}}

\item{from}{Time series start date.}

\item{to}{Time series end date.}

\item{time_type}{If "auto", \code{periods} are used for
the time expansion when days, weeks, months or years are specified,
and \code{durations}
are used otherwise.}

\item{include_interval}{Logical. If \code{TRUE} then
a column "interval" of the form \verb{time_min <= x < time_max} is added
showing the time interval in which the respective counts belong to.
The rightmost interval will always be closed.}

\item{.by}{(Optional). A selection of columns to group by for this operation.
Columns are specified using tidy-select.}

\item{time_floor}{Should \code{from} be floored to the nearest unit
specified through the \code{time_by}
argument? This is particularly useful for starting
sequences at the beginning of a week
or month for example.}

\item{week_start}{day on which week starts following ISO conventions - 1
means Monday, 7 means Sunday (default).
This is only used when \code{time_floor = TRUE}.}

\item{roll_month}{Control how impossible dates are handled when
month or year arithmetic is involved.
Options are "preday", "boundary", "postday", "full" and "NA".
See \code{?timechange::time_add} for more details.}

\item{roll_dst}{See \code{?timechange::time_add} for the full list of details.}

\item{sort}{Should the result be sorted? Default is \code{TRUE}.
If \code{FALSE} then original (input) order is kept.
The sorting only applies to groups and time variable.}
}
\value{
A summarised \code{data.frame}.
}
\description{
This works much the same as \code{dplyr::summarise()}, except that
you can supply an additional \code{time} argument to allow for
aggregating time to a higher unit.
}
\examples{
library(timeplyr)
library(dplyr)
library(lubridate)
library(nycflights13)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
# Works the same way as summarise()
# Monthly average arrival time
flights \%>\%
  mutate(date = as_date(time_hour)) \%>\%
  time_summarise(mean_arr_time = mean(arr_time, na.rm = TRUE),
                 time = date,
                 time_by = "month",
                 include_interval = TRUE)
# Example of monthly summary using zoo's yearmon
\donttest{
flights \%>\%
  mutate(yearmon = zoo::as.yearmon(as_date(time_hour))) \%>\%
  time_summarise(time = yearmon,
                 n = n(),
                 mean_arr_time = mean(arr_time, na.rm = TRUE),
                 include_interval = TRUE)
}
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
