% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/upsample.R
\name{step_upsample}
\alias{step_upsample}
\alias{tidy.step_upsample}
\title{Up-Sample a Data Set Based on a Factor Variable}
\usage{
step_upsample(
  recipe,
  ...,
  over_ratio = 1,
  ratio = NA,
  role = NA,
  trained = FALSE,
  column = NULL,
  target = NA,
  skip = TRUE,
  seed = sample.int(10^5, 1),
  id = rand_id("upsample")
)

\method{tidy}{step_upsample}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variable is used to sample the data. See \code{\link[=selections]{selections()}}
for more details. The selection should result in \emph{single
factor variable}. For the \code{tidy} method, these are not
currently used.}

\item{over_ratio}{A numeric value for the ratio of the
majority-to-minority frequencies. The default value (1) means
that all other levels are sampled up to have the same
frequency as the most occurring level. A value of 0.5 would mean
that the minority levels will have (at most) (approximately)
half as many rows than the majority level.}

\item{ratio}{Deprecated argument; same as \code{over_ratio}.}

\item{role}{Not used by this step since no new variables are
created.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{column}{A character string of the variable name that will
be populated (eventually) by the \code{...} selectors.}

\item{target}{An integer that will be used to subsample. This
should not be set by the user and will be populated by \code{prep}.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[recipes:bake.recipe]{bake.recipe()}}? While all operations are baked
when \code{\link[recipes:prep.recipe]{prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations}

\item{seed}{An integer that will be used as the seed when upsampling.}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_upsample} object.}
}
\value{
An updated version of \code{recipe} with the new step
added to the sequence of existing steps (if any). For the
\code{tidy} method, a tibble with columns \code{terms} which is
the variable used to sample.
}
\description{
\code{step_upsample} creates a \emph{specification} of a recipe step that
will replicate rows of a data set to make the occurrence of
levels in a specific factor level equal.
}
\details{
Up-sampling is intended to be performed on the \emph{training} set
alone. For this reason, the default is \code{skip = TRUE}. It is
advisable to use \code{prep(recipe, retain = TRUE)} when preparing
the recipe; in this way \code{\link[=juice]{juice()}} can be used to obtain the
up-sampled version of the data.

If there are missing values in the factor variable that is used
to define the sampling, missing data are selected at random in
the same way that the other factor levels are sampled. Missing
values are not used to determine the amount of data in the
majority level (see example below).

For any data with factor levels occurring with the same
frequency as the majority level, all data will be retained.

All columns in the data are sampled and returned by \code{\link[=juice]{juice()}}
and \code{\link[=bake]{bake()}}.
}
\examples{
library(recipes)
library(modeldata)
data(okc)

orig <- table(okc$diet, useNA = "always")

sort(orig, decreasing = TRUE)

up_rec <- recipe( ~ ., data = okc) \%>\%
  # Bring the minority levels up to about 200 each
  # 200/16562 is approx 0.0121
  step_upsample(diet, over_ratio = 0.0121) \%>\%
  prep(training = okc, retain = TRUE)

training <- table(juice(up_rec)$diet, useNA = "always")

# Since `skip` defaults to TRUE, baking the step has no effect
baked_okc <- bake(up_rec, new_data = okc)
baked <- table(baked_okc$diet, useNA = "always")

# Note that if the original data contained more rows than the
# target n (= ratio * majority_n), the data are left alone:
data.frame(
  level = names(orig),
  orig_freq = as.vector(orig),
  train_freq = as.vector(training),
  baked_freq = as.vector(baked)
)

library(ggplot2)

ggplot(circle_example, aes(x, y, color = class)) +
  geom_point() +
  labs(title = "Without upsample")

recipe(class ~ ., data = circle_example) \%>\%
  step_nearmiss(class) \%>\%
  prep() \%>\%
  juice() \%>\%
  ggplot(aes(x, y, color = class)) +
  geom_jitter() +
  labs(title = "With upsample (with jittering)")
}
\concept{preprocessing}
\concept{subsampling}
\keyword{datagen}
