% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table1.R
\name{table1}
\alias{table1}
\alias{table1.default}
\alias{table1.formula}
\title{Generate an HTML table of descriptive statistics.}
\usage{
table1(x, ...)

\method{table1}{default}(x, labels, groupspan = NULL, rowlabelhead = "",
  transpose = FALSE, topclass = "Rtable1", footnote = NULL,
  render = render.default, ...)

\method{table1}{formula}(x, data, overall = "Overall", rowlabelhead = "",
  transpose = FALSE, droplevels = TRUE, topclass = "Rtable1",
  footnote = NULL, render = render.default, ...)
}
\arguments{
\item{x}{An object, typically a \code{formula} or list of \code{data.frame}s.}

\item{...}{Further arguments, passed to \code{render}.}

\item{labels}{A list containing labels for variables, strata and groups (see Details).}

\item{groupspan}{A vector of integers specifying the number of strata to group together.}

\item{rowlabelhead}{A heading for the first column of the table, which contains the row labels.}

\item{transpose}{Logical. Should the table be transposed (i.e. strata as
rows and variables as columns)?}

\item{topclass}{A class attribute for the outermost (i.e. \code{<table>}) tag.}

\item{footnote}{A character string to be added as a footnote to the table.
The default \code{NULL} causes the footnote to be omitted.}

\item{render}{A function to render the table cells (see Details).
immediately displayed? Otherwise an HTML fragment is printed to
\code{\link{stdout}}.}

\item{data}{For the formula interface, a \code{data.frame} from which the
variables in \code{x} should be taken.}

\item{overall}{A label for the "Overall" column. Specify \code{NULL} or
\code{FALSE} to omit the column altogether.}

\item{droplevels}{Should empty factor levels be dropped?}
}
\value{
An object of class "table1".
}
\description{
There are two interfaces, the default, which typically takes a list of
\code{data.frame}s for \code{x}, and the formula interface. The formula
interface is less flexible, but simpler to use and designed to handle the
most common use cases. It is important to use factors appropriately for
categorical variables (i.e. have the levels labeled properly and in the
desired order). The contents of the table can be customized by providing
user-defined `renderer' functions. Customization of the table appearance is
deliberately not attempted, as this is best accomplished with CSS. To
facilitate this, some tags (such as row labels) are given specific classes
for easy CSS selection.
}
\details{
For the default version, is is expected that \code{x} is a named
list of \code{data.frame}s, one for each stratum, with names corresponding to
strata labels.
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: The default interface, where \code{x} is a \code{data.frame}.

\item \code{formula}: The \code{formula} interface.
}}

\examples{

dat <- expand.grid(id=1:10, sex=c("Male", "Female"), treat=c("Treated", "Placebo"))
dat$age <- runif(nrow(dat), 10, 50)
dat$age[3] <- NA  # Add a missing value
dat$wt <- exp(rnorm(nrow(dat), log(70), 0.2))

label(dat$sex) <- "Sex"
label(dat$age) <- "Age"
label(dat$treat) <- "Treatment Group"
label(dat$wt) <- "Weight"

units(dat$age) <- "years"
units(dat$wt) <- "kg"

# One level of stratification
table1(~ sex + age + wt | treat, data=dat)

# Two levels of stratification (nesting)
table1(~ age + wt | treat*sex, data=dat)

# Switch the order or nesting
table1(~ age + wt | sex*treat, data=dat)

# No stratification
table1(~ treat + sex + age + wt, data=dat)

# Something more complicated

dat$dose <- ifelse(dat$treat=="Placebo", "Placebo",
                   sample(c("5 mg", "10 mg"), nrow(dat), replace=TRUE))
dat$dose <- factor(dat$dose, levels=c("Placebo", "5 mg", "10 mg"))

strata <- c(split(dat, dat$dose),
            list("All treated"=subset(dat, treat=="Treated")),
            list(Overall=dat))

labels <- list(
    variables=list(sex=render.varlabel(dat$sex),
                   age=render.varlabel(dat$age),
                   wt=render.varlabel(dat$wt)),
    groups=list("", "Treated", ""))

my.render.cont <- function(x) {
    with(stats.default(x), 
        sprintf("\%0.2f (\%0.1f)", MEAN, SD))
}

table1(strata, labels, groupspan=c(1, 3, 1), render.continuous=my.render.cont)

# Transposed table
table1(~ age + wt | treat, data=dat, transpose=TRUE)

}
\keyword{utilities}
