\name{simdata_guo}
\alias{simdata_guo}
\title{Generates data from \code{K} multivariate normal data populations having the
covariance structure from Guo et al. (2007).}
\usage{
  simdata_guo(n, mean, block_size, num_blocks, rho,
    sigma2 = 1, seed = NULL)
}
\arguments{
  \item{n}{a vector (of length K) of the sample sizes for
  each population}

  \item{mean}{a vector or a list (of length K) of mean
  vectors}

  \item{block_size}{a vector (of length K) of the sizes of
  the square block matrices for each population. See
  details.}

  \item{num_blocks}{a vector (of length K) giving the
  number of block matrices for each population. See
  details.}

  \item{rho}{a vector (of length K) of the values of the
  autocorrelation parameter for each class covariance
  matrix}

  \item{sigma2}{a vector (of length K) of the variance
  coefficients for each class covariance matrix}

  \item{seed}{seed for random number generation (If
  \code{NULL}, does not set seed)}
}
\value{
  named list containing: \describe{ \item{x:}{A matrix
  whose rows are the observations generated and whose
  columns are the \code{p} features (variables)}
  \item{y:}{A vector denoting the population from which the
  observation in each row was generated.} }
}
\description{
  We generate \eqn{n_k} observations \eqn{(k = 1, \ldots,
  K)} from each of \eqn{K} multivariate normal
  distributions. Let the \eqn{k}th population have a
  \eqn{p}-dimensional multivariate normal distribution,
  \eqn{N_p(\mu_k, \Sigma_k)} with mean vector \eqn{\mu_k}
  and positive-definite covariance matrix \eqn{\Sigma_k}.
  Each covariance matrix \eqn{\Sigma_k} consists of
  block-diagonal autocorrelation matrices.
}
\details{
  The \eqn{k}th class covariance matrix is defined as
  \deqn{\Sigma_k = \Sigma^{(\rho)} \oplus \Sigma^{(-\rho)}
  \oplus \ldots \oplus \Sigma^{(\rho)},} where \eqn{\oplus}
  denotes the direct sum and the \eqn{(i,j)}th entry of
  \eqn{\Sigma^{(\rho)}} is \deqn{\Sigma_{ij}^{(\rho)} = \{
  \rho^{|i - j|} \}.}

  The matrix \eqn{\Sigma^{(\rho)}} is referred to as a
  block. Its dimensions are provided in the
  \code{block_size} argument, and the number of blocks are
  specified in the \code{num_blocks} argument.

  Each matrix \eqn{\Sigma_k} is generated by the
  \code{\link{cov_block_autocorrelation}} function.

  The number of populations, \code{K}, is determined from
  the length of the vector of sample sizes, code{n}. The
  mean vectors can be given in a list of length \code{K}.
  If one mean is given (as a vector or a list having 1
  element), then all populations share this common mean.

  The block sizes can be given as a numeric vector or a
  single value, in which case the degrees of freedom is
  replicated \code{K} times. The same logic applies to
  \code{num_blocks}, \code{rho}, and \code{sigma2}.

  For each class, the number of features, \code{p}, is
  computed as \code{block_size * num_blocks}. The values
  for \code{p} must agree for each class.

  The block-diagonal covariance matrix with autocorrelated
  blocks was popularized by Guo et al. (2007) for studying
  classification of high-dimensional data.
}
\examples{
# Generates 10 observations from two multivariate normal populations having
# a block-diagonal autocorrelation structure.
block_size <- 3
num_blocks <- 3
p <- block_size * num_blocks
means_list <- list(seq_len(p), -seq_len(p))
data <- simdata_guo(n = c(10, 10), mean = means_list, block_size = block_size,
                    num_blocks = num_blocks, rho = 0.9, seed = 42)
dim(data$x)
table(data$y)

# Generates 15 observations from each of three multivariate normal
# populations having block-diagonal autocorrelation structures. The
# covariance matrices are unequal.
p <- 16
block_size <- c(2, 4, 8)
num_blocks <- p / block_size
rho <- c(0.1, 0.5, 0.9)
sigma2 <- 1:3
mean_list <- list(rep.int(-5, p), rep.int(0, p), rep.int(5, p))

set.seed(42)
data2 <- simdata_guo(n = c(15, 15, 15), mean = mean_list,
                    block_size = block_size, num_blocks = num_blocks,
                    rho = rho, sigma2 = sigma2)
dim(data2$x)
table(data2$y)
}
\references{
  Guo, Y., Hastie, T., & Tibshirani, R. (2007).
  "Regularized linear discriminant analysis and its
  application in microarrays," Biostatistics, 8, 1, 86-100.
}

