% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utility.R
\name{smog.default}
\alias{smog.default}
\alias{smog.formula}
\title{Generalized linear model constraint on hierarchical structure
by using overlapped group penalty}
\usage{
\method{smog}{default}(x, y, g, v, label, lambda1, lambda2, lambda3,
  family = "gaussian", subset = NULL, rho = 10, scale = TRUE,
  eabs = 0.001, erel = 0.001, LL = 1, eta = 1.25, maxitr = 1000, ...)

\method{smog}{formula}(formula, data = list(), g, v, label, lambda1, lambda2,
  lambda3, ...)
}
\arguments{
\item{x}{a model matrix, or a data frame of dimensions n by p, 
in which the columns represents the predictor variables.}

\item{y}{response variable, corresponds to the family description. 
When family is ``gaussian'' or ``binomial'', \code{y} ought to
be a numeric vector of observations of length n; when family 
is ``coxph'', \code{y} represents the survival objects, containing the 
survival time and the censoring status. See \code{\link[survival]{Surv}}.}

\item{g}{a vector of group labels for the predictor variables.}

\item{v}{a vector of binary values, represents whether or not the 
predictor variables are penalized. Note that 1 indicates 
penalization and 0 for not penalization.}

\item{label}{a character vector, represents the type of predictors in terms of treatment,
prognostic, and predictive effects by using ``t'', ``prog'', and ``pred'',
respectively.}

\item{lambda1}{penalty parameter for the L2 norm of each group of prognostic and predictive effects.}

\item{lambda2}{ridge penalty parameter for the squared L2 norm of each group of prognostic and predictive effects.}

\item{lambda3}{penalty parameter for the L1 norm of predictive effects.}

\item{family}{a description of the distribution family for the response 
variable variable. For continuous response variable,
family is ``gaussian''; for multinomial or binary response
variable, family is ``binomial''; for survival response
variable, family is ``coxph'', respectively.}

\item{subset}{an optional vector specifying a subset of observations to be 
used in the model fitting. Default is \code{NULL}.}

\item{rho}{the penalty parameter used in the alternating direction method 
of multipliers (ADMM) algorithm. Default is 10.}

\item{scale}{whether or not scale the design matrix. Default is \code{TRUE}.}

\item{eabs}{the absolute tolerance used in the ADMM algorithm. Default is 1e-3.}

\item{erel}{the reletive tolerance used in the ADMM algorithm. Default is 1e-3.}

\item{LL}{initial value for the Lipschitz continuous constant for 
approximation to the objective function in the Majorization-
Minimization (MM) (or iterative shrinkage-thresholding algorithm 
(ISTA)). Default is 1.}

\item{eta}{gradient stepsize for the backtrack line search for the Lipschitz
continuous constant. Default is 1.25.}

\item{maxitr}{the maximum iterations for convergence in the ADMM algorithm. 
Default is 1000.}

\item{...}{other relevant arguments that can be supplied to smog.}

\item{formula}{an object of class ``formula'': a symbolic description of the
model to be fitted. Should not include the intercept.}

\item{data}{an optional data frame, containing the variables in the model.}
}
\value{
\code{smog} returns an object of class inhering from ``smog''. The 
        generic accessor functions \code{coef}, \code{coefficients}, 
        \code{fitted.value}, and \code{predict} can be used to extract
        various useful features of the value returned by \code{smog}.
        
        An object of ``smog'' is a list containing at least the following 
        components: 
        
        \item{coefficients}{a data frame containing the nonzero predictor
                            variables' indexes, names, and estimates. When
                            family is ``binomial'', the estimates have K-1 
                            columns, each column representing the weights for the 
                            corresponding group. The last group behaves the
                            ``pivot''.}
        \item{fitted.values}{the fitted mean values for the response variable,
                             for family is ``gaussian''. When family is 
                             ``binomial", the fitted.values are the probabilies
                             for each class; when family is ``coxph'', 
                             the fitted.values are survival probabilities.}
        \item{model}{a list of estimates for the intercept, treatment effect, 
                     and prognostic and predictive effects for the selectd
                     biomarkers.}
        \item{weight}{the weight of predictors resulted from the penalty funciton,
                      is used to calculate the degrees of freedom.}
        \item{DF}{the degrees of freedom. When family = ``gaussian'', 
                  \eqn{DF = tr(x_{\lambda}'(x_{\lambda}'x_{\lambda}+W)x_{\lambda})}. 
                  For other families, DF is approximated by \eqn{diag(1/(1+W))}.}
        \item{criteria}{model selection criteria, including the correction Akaike's Information 
                        Criterion (AIC), AIC, Bayesian Information Criterion (BIC), and the generalized 
                        cross-validation score (GCV), respectively. See also \code{\link{cv.smog}}.
                        \describe{
                        \item{cAIC}{\eqn{\frac{n}{2}}log(|2*log-likelihood|) + 
                                    \eqn{\frac{n}{2} \left( \frac{1+k/n}{1-k+2/n} \right)}.}
                        \item{AIC}{log(|2*log-likelihood|/n) + \eqn{2\frac{k}{n}}.}
                        \item{BIC}{log(|2*log-likelihood|/n) + \eqn{2\frac{k}{n}}log(n).}
                        \item{GCV}{|2*log-likelihood| / \eqn{(n(1-k/n)^2)}.}
                        }
                        Where k is the degrees of freedom \code{DF}, which is related to the 
                        penalty parameters \eqn{\lambda}'s. 
                        }
        \item{llikelihood}{the log-likelihood value for the converged model.}
        \item{loglike}{the penalized log-likelihood values for each 
                       iteration in the algorithm.}
        \item{PrimalError}{the averged norms \eqn{||\beta-Z||/\sqrt{p}} for each iteration,
                           in the ADMM algorithm.}
        \item{DualError}{the averaged norms \eqn{||Z^{t+1}-Z^{t}||/\sqrt{p}} for 
                         each iteration, in the ADMM algorithm.}
        \item{converge}{the number of iterations processed in the ADMM algorithm.}
        \item{call}{the matched call.}
        \item{formula}{the formula supplied.}
}
\description{
\code{smog} fits a linear non-penalized phynotype (demographic) variables such as 
age, gender, treatment, etc, and penalized groups of prognostic effect (main effect)
and predictive effect (interaction effect), by satisfying the hierarchy structure:
if a predictive effect exists, its prognostic effect must be in the model. It can deal
with continuous, binomial or multinomial, and survival response variables, underlying 
the assumption of Gaussian, binomial (multinomial), and Cox proportional hazard models,
respectively. It can accept \code{\link[stats]{formula}}, and output coefficients table,
fitted.values, and convergence information produced in the algorithm iterations.
}
\details{
The formula has the form \code{response ~ 0 + terms} where \code{terms} is
         a series of predictor variables to be fitted for \code{response}. For \code{gaussian} 
         family, the response is a continuous vector. For \code{binomial} family, 
         the response is a factor vector, in which the last level denotes the ``pivot''.
         For \code{coxph} family, the response is a \code{\link[survival]{Surv}} 
         object, containing the survival time and censoring status.
         
         The terms contains the non-penalized predictor variables, and many groups 
         of prognostic and predictive terms, where in each group the prognostic 
         term comes first, followed by the predictive term.
         
         Different hierachical structures within groups can result from adjusting 
         the penalty parameters in the penalty function:          
         \deqn{\Omega(\bm{\beta}) = \lambda_1||\bm{\beta}|| +
         \lambda_2||\bm{\beta}||^2+\lambda_3|\beta_2|}
         Where \eqn{\bm{\beta}=(\beta_1,\beta_2)}. Note that \eqn{\beta_1} denotes
         the prognostic effect (main effect), and \eqn{\beta_2} for the predictive effect 
         (interactive effect), respectively. When \eqn{\lambda_2 = 0}
         and \eqn{\lambda_3 = 0}, it indicates no structure within groups.
         When \eqn{\lambda_2 \ne 0}, the penalty function honors the structure within groups
         such that: predictive effect \eqn{\ne 0 \Longrightarrow} prognostic effect \eqn{\ne 0}. 
         
         \code{rho,eabs,erel,LL,eta} are the corresponding parameters used in the itervative
         shrinkage-thresholding algorithm (ISTA) and the alternating direction method of 
         multipliers algorithm (ADMM). 
         
         Note that the missing values in the data are supposed to be dealt with in the 
         data preprocessing, before applying the method.
}
\examples{
 
require(coxed)

n=50;p=100
set.seed(2018)
# generate design matrix x
s=10
x=matrix(0,n,1+2*p)
x[,1]=sample(c(0,1),n,replace = TRUE)
x[,seq(2,1+2*p,2)]=matrix(rnorm(n*p),n,p)
x[,seq(3,1+2*p,2)]=x[,seq(2,1+2*p,2)]*x[,1]

g=c(p+1,rep(1:p,rep(2,p)))  # groups 
v=c(0,rep(1,2*p))           # penalization status
label=c("t",rep(c("prog","pred"),p))  # type of predictor variables

# generate beta
beta=c(rnorm(13,0,2),rep(0,ncol(x)-13))
beta[c(2,4,7,9)]=0

# generate y
data1=x\%*\%beta
noise1=rnorm(n)
snr1=as.numeric(sqrt(var(data1)/(s*var(noise1))))
y1=data1+snr1*noise1
lfit1=smog(x,y1,g,v,label,lambda1=8,lambda2=0,lambda3=8,family = "gaussian")

## generate binomial data
prob=exp(as.matrix(x)\%*\%as.matrix(beta))/(1+exp(as.matrix(x)\%*\%as.matrix(beta)))
y2=ifelse(prob<0.5,0,1)
lfit2=smog(x,y2,g,v,label,lambda1=0.03,lambda2=0,lambda3=0.03,family = "binomial")

## generate survival data
data3=sim.survdata(N=n,T=100,X=x,beta=beta)
y3=data3$data[,c("y","failed")]
y3$failed=ifelse(y3$failed,1,0)
colnames(y3)=c("time","status")
lfit3=smog(x,y3,g,v,label,lambda1=0.2,lambda2=0,lambda3=0.2,family = "coxph")

}
\references{
\insertRef{ma2019structural}{smog}
}
\seealso{
\code{\link{cv.smog}}, \code{\link{predict.smog}}, \code{\link{plot.smog}}.
}
\author{
Chong Ma, \email{chongma8903@gmail.com}.
}
