#' @title Tidy summary output for stan models
#' @name tidy_stan
#'
#' @description Returns a tidy summary output for stan models.
#'
#' @param x A \code{stanreg}, \code{stanfit} or \code{brmsfit} object.
#' @param typical The typical value that will represent the Bayesian point estimate.
#'        By default, the posterior median is returned. See \code{\link{typical_value}}
#'        for possible values for this argument.
#' @param trans Name of a function or character vector naming a function, used
#'        to apply transformations on the estimate and HDI-values. The
#'        values for standard errors are \emph{not} transformed!
#' @param digits Amount of digits to round numerical values in the output.
#'
#' @inheritParams hdi
#'
#' @return A tidy data frame, summarizing \code{x}, with consistent column names.
#'         To distinguish multiple HDI values, column names for the HDI get a suffix
#'         when \code{prob} has more than one element.
#'
#' @details The returned data frame gives information on the Bayesian point
#'          estimate (column \emph{estimate}, which is by default the posterior
#'          median; other statistics are also possible, see \code{typical}), the
#'          standard error (which are actually \emph{median absolute deviations}),
#'          the HDI, the ratio of effective numbers of samples, \emph{n_eff},
#'          (i.e. effective number of samples divided by total number of samples)
#'          and Rhat statistics.
#'          \cr \cr
#'          The ratio of effective number of samples ranges from 0 to 1,
#'          and should be close to 1. The closer this ratio comes to zero means
#'          that the chains may be inefficient, but possibly still okay.
#'          \cr \cr
#'          When Rhat is above 1, it usually indicates that the chain has not
#'          yet converged, indicating that the drawn samples might not be
#'          trustworthy. Drawing more iteration may solve this issue.
#'          \cr \cr
#'          Computation for HDI is based on the code from Kruschke 2015, pp. 727f.
#'
#' @seealso \code{\link{hdi}}
#'
#' @references Kruschke JK. \emph{Doing Bayesian Data Analysis: A Tutorial with R, JAGS, and Stan.} 2nd edition. Academic Press, 2015
#' \cr \cr
#' Gelman A, Carlin JB, Stern HS, Dunson DB, Vehtari A, Rubin DB. \emph{Bayesian data analysis.} 3rd ed. Boca Raton: Chapman & Hall/CRC, 2013
#' \cr \cr
#' Gelman A, Rubin DB. \emph{Inference from iterative simulation using multiple sequences.} Statistical Science 1992;7: 457–511
#' \cr \cr
#' McElreath R. \emph{Statistical Rethinking. A Bayesian Course with Examples in R and Stan.} Chapman and Hall, 2015
#'
#' @examples
#' \dontrun{
#' if (require("rstanarm")) {
#'   fit <- stan_glm(mpg ~ wt + am, data = mtcars, chains = 1)
#'   tidy_stan(fit)
#'   tidy_stan(fit, prob = c(.89, .5))
#' }}
#'
#' @importFrom purrr map flatten_dbl map_dbl modify_if
#' @importFrom dplyr bind_cols select mutate
#' @importFrom tidyselect starts_with
#' @importFrom tibble add_column
#' @importFrom stats mad
#' @importFrom bayesplot rhat neff_ratio
#' @export
tidy_stan <- function(x, prob = .89, typical = "median", trans = NULL, type = c("fixed", "random", "all"), digits = 3) {

  # only works for rstanarm- or brms-models
  if (!inherits(x, c("stanreg", "stanfit", "brmsfit")))
    stop("`x` needs to be a stanreg- or brmsfit-object.", call. = F)

  # check arguments
  type <- match.arg(type)

  # get data frame
  mod.dat <- as.data.frame(x)

  # for brmsfit models, we need to remove some columns here to
  # match data rows later
  if (inherits(x, "brmsfit")) mod.dat <- brms_clean(mod.dat)

  # compute HDI
  out <- hdi(x, prob = prob, trans = trans, type = "all")

  # we need names of elements, for correct removal
  nr <- bayesplot::neff_ratio(x)

  if (inherits(x, "brmsfit")) {
    cnames <- make.names(names(nr))
    keep <- cnames %in% out$term
  } else {
    keep <- 1:nrow(out)
  }


  # compute additional statistics, like point estimate, standard errors etc.

  nr <- nr[keep]
  rh <- bayesplot::rhat(x)[keep]
  se <- dplyr::pull(mcse(x, type = "all"), "mcse")[keep]


  out <- out %>%
    tibble::add_column(
      estimate = purrr::map_dbl(mod.dat, ~ typical_value(.x, fun = typical)),
      std.error = purrr::map_dbl(mod.dat, stats::mad),
      .after = 1
    ) %>%
    dplyr::mutate(
      n_eff = nr,
      Rhat = rh,
      mcse = se
    )


  # transform estimate, if requested

  if (!is.null(trans)) {
    trans <- match.fun(trans)
    out$estimate <- trans(out$estimate)
  }


  # check if we need to remove random or fixed effects
  out <- remove_effects_from_stan(out, type, is.brms = inherits(x, "brmsfit"))

  # round values
  purrr::modify_if(out, is.numeric, ~ round(.x, digits = digits))
}


#' @importFrom tidyselect starts_with ends_with
#' @importFrom dplyr slice
#' @importFrom tibble as_tibble
#' @importFrom sjmisc is_empty
remove_effects_from_stan <- function(out, type, is.brms) {

  # brmsfit-objects also include sd and cor for mixed
  # effecs models, so remove these here

  if (is.brms) out <- brms_clean(out)


  # remove certain terms like log-posterior etc. from output

  keep <-  which(!(out$term %in% c("lp__", "log-posterior", "mean_PPD")))
  out <- dplyr::slice(out, !! keep)


  # if user wants all terms, return data here

  if (type == "all") return(tibble::as_tibble(out))


  # check if we need to remove random or fixed effects
  # therefor, find random effect parts first

  re <- tidyselect::starts_with("b[", vars = out$term)
  re.s <- tidyselect::starts_with("Sigma[", vars = out$term)
  re.i <- intersect(
    tidyselect::starts_with("r_", vars = out$term),
    tidyselect::ends_with(".", vars = out$term)
  )

  removers <- unique(c(re, re.s, re.i))

  if (!sjmisc::is_empty(removers)) {
    if (type == "fixed") {
      # remove all random effects
      out <- dplyr::slice(out, !! -removers)
    } else if (type == "random") {
      # remove all fixed effects
      out <- dplyr::slice(out, !! removers)
    }
  }


  tibble::as_tibble(out)
}


#' @importFrom tidyselect starts_with ends_with
#' @importFrom dplyr slice
#' @importFrom sjmisc is_empty
brms_clean <- function(out) {

  # brmsfit-objects also include sd and cor for mixed
  # effecs models, so remove these here

  if (tibble::has_name(out, "term")) {
    re.sd <- tidyselect::starts_with("sd_", vars = out$term)
    re.cor <- tidyselect::starts_with("cor_", vars = out$term)
    lp <- tidyselect::starts_with("lp__", vars = out$term)

    removers <- unique(c(re.sd, re.cor, lp))

    if (!sjmisc::is_empty(removers))
      out <- dplyr::slice(out, !! -removers)
  }


  # we may have transformed data frame, where columns
  # need to be removed

  re.sd <- tidyselect::starts_with("sd_", vars = colnames(out))
  re.cor <- tidyselect::starts_with("cor_", vars = colnames(out))
  lp <- tidyselect::starts_with("lp__", vars = colnames(out))

  removers <- unique(c(re.sd, re.cor, lp))

  if (!sjmisc::is_empty(removers))
    out <- dplyr::select(out, !! -removers)

  out
}
