% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sjPlotInteractions.R
\name{sjp.int}
\alias{sjp.int}
\title{Plot interaction effects of (generalized) linear (mixed) models}
\usage{
sjp.int(fit, type = c("eff", "cond", "emm"), int.term = NULL,
  int.plot.index = NULL, mdrt.values = c("minmax", "meansd", "zeromax",
  "quart", "all"), swap.pred = FALSE, plevel = 0.1, diff = FALSE,
  title = NULL, axis.title = NULL, axis.labels = NULL,
  legend.title = NULL, legend.labels = NULL, wrap.title = 50,
  wrap.legend.labels = 20, wrap.legend.title = 20, geom.colors = "Set1",
  geom.size = NULL, fill.color = "grey", fill.alpha = 0.3,
  show.values = FALSE, show.ci = FALSE, p.kr = TRUE, grid.breaks = NULL,
  xlim = NULL, ylim = NULL, y.offset = 0.07, digits = 2,
  facet.grid = FALSE, prnt.plot = TRUE, ...)
}
\arguments{
\item{fit}{the fitted (generalized) linear (mixed) model object, including interaction terms. Accepted model
classes are
\itemize{
  \item linear models (\code{\link{lm}})
  \item generalized linear models (\code{\link{glm}})
  \item linear mixed effects models (\code{\link[lme4]{lmer}})
  \item generalized linear mixed effects models (\code{\link[lme4]{glmer}})
  \item non-linear mixed effects models (\code{\link[lme4]{nlmer}})
  \item linear mixed effects models (\code{\link[nlme]{lme}}, but only for \code{type = "eff"})
  \item generalized least squares models (\code{\link[nlme]{gls}}, but only for \code{type = "eff"})
  \item panel data estimators (\code{\link[plm]{plm}})
}}

\item{type}{interaction plot type. Use one of following values:
\describe{
  \item{\code{type = "eff"}}{(default) plots the overall moderation effect on the response value. See 'Details'.}
  \item{\code{type = "cond"}}{plots the mere \emph{change} of the moderating effect on the response value (conditional effect). See 'Details'.}
  \item{\code{type = "emm"}}{plots the estimated marginal means (least square means). If this type is chosen, not all function arguments are applicable. See 'Details'.}
}}

\item{int.term}{name of interaction term of \code{fit} (as character), which should be plotted
when using \code{type = "eff"}. By default, this argument will be ignored
(i.e. \code{int.term = NULL}). See 'Details'.}

\item{int.plot.index}{numeric vector with index numbers that indicate which 
interaction terms should be plotted in case the \code{fit} has more than
one interaction. By default, this value is \code{NULL}, hence all interactions
are plotted.}

\item{mdrt.values}{indicates which values of the moderator variable should be 
used when plotting the interaction effects.
\describe{
  \item{\code{"minmax"}}{(default) minimum and maximum values (lower and upper bounds) of the moderator are used to plot the interaction between independent variable and moderator.}
  \item{\code{"meansd"}}{uses the mean value of the moderator as well as one standard deviation below and above mean value to plot the effect of the moderator on the independent variable (following the convention suggested by Cohen and Cohen and popularized by Aiken and West, i.e. using the mean, the value one standard deviation above, and the value one standard deviation below the mean as values of the moderator, see \href{http://www.theanalysisfactor.com/3-tips-interpreting-moderation/}{Grace-Martin K: 3 Tips to Make Interpreting Moderation Effects Easier}).}
  \item{\code{"zeromax"}}{is similar to the \code{"minmax"} option, however, \code{0} is always used as minimum value for the moderator. This may be useful for predictors that don't have an empirical zero-value, but absence of moderation should be simulated by using 0 as minimum.}
  \item{\code{"quart"}}{calculates and uses the quartiles (lower, median and upper) of the moderator value.}
  \item{\code{"all"}}{uses all values of the moderator variable. Note that this option only applies to \code{type = "eff"}, for numeric moderator values.}
}}

\item{swap.pred}{if \code{TRUE}, the predictor on the x-axis and the moderator value in an interaction are
swapped. For \code{type = "eff"}, the first interaction term is used as moderator and the second term
is plotted at the x-axis. For \code{type = "cond"}, the interaction's predictor with less unique values is 
printed along the x-axis. Default is \code{FALSE}, so the second predictor in an interaction, respectively 
the predictor with more unique values is printed along the x-axis.}

\item{plevel}{indicates at which p-value an interaction term is considered as \emph{significant},
i.e. at which p-level an interaction term will be considered for plotting. Default is
0.1 (10 percent), hence, non-significant interactions are excluded by default. This
argument does not apply to \code{type = "eff"}.}

\item{diff}{if \code{FALSE} (default), the minimum and maximum interaction effects of the moderating variable
is shown (one line each). if \code{TRUE}, only the difference between minimum and maximum interaction effect
is shown (single line). Only applies to \code{type = "cond"}.}

\item{title}{default title used for the plots. Should be a character vector
of same length as interaction plots to be plotted. Default value is \code{NULL}, which means that each plot's title
includes the dependent variable as well as the names of the interaction terms.}

\item{axis.title}{a default title used for the x-axis. Should be a character vector
of same length as interaction plots to be plotted. Default value is \code{NULL},
which means that each plot's x-axis uses the predictor's name as title.}

\item{axis.labels}{character vector with value labels of the interaction, used
to label the x-axis. Only applies to \code{type = "emm"}.}

\item{legend.title}{title of the diagram's legend. A character vector of same length as 
amount of interaction plots to be plotted (i.e. one vector element for each
plot's legend title).}

\item{legend.labels}{labels for the guide/legend. Either a character vector of same length as
amount of legend labels of the plot, or a \code{list} of character vectors, if more than one
interaction plot is plotted (i.e. one vector of legend labels for each interaction plot).
Default is \code{NULL}, so the name of the predictor with min/max-effect is used 
as legend label.}

\item{wrap.title}{numeric, determines how many chars of the plot title are displayed in
one line and when a line break is inserted.}

\item{wrap.legend.labels}{numeric, determines how many chars of the legend labels are 
displayed in one line and when a line break is inserted.}

\item{wrap.legend.title}{numeric, determines how many chars of the legend's title 
are displayed in one line and when a line break is inserted.}

\item{geom.colors}{vector of color values or name of a valid color brewer palette. 
If not a color brewer palette name, \code{geom.colors} must be of same 
length as moderator values used in the plot (see \code{mdrt.values}).
See also 'Details' in \code{\link{sjp.grpfrq}}.}

\item{geom.size}{size resp. width of the geoms (bar width, line thickness or point size, 
depending on plot type and function). Note that bar and bin widths mostly 
need smaller values than dot sizes.}

\item{fill.color}{fill color of the shaded area between the minimum and maximum lines. Default is \code{"grey"}.
Either set \code{fill.color} to \code{NULL} or use 0 for \code{fill.alpha} if you want to hide the shaded area.}

\item{fill.alpha}{alpha value (transparancy) of the shaded area between the minimum and maximum lines. Default is 0.4.
Use either 0 or set \code{fill.color} to \code{NULL} if you want to hide the shaded area.}

\item{show.values}{logical, whether values should be plotted or not.}

\item{show.ci}{may be a numeric or logical value. If \code{show.ci} is logical and 
\code{TRUE}, a 95\% confidence region will be plotted. If \code{show.ci}
if numeric, must be a number between 0 and 1, indicating the proportion
for the confidence regeion (e.g. \code{show.ci = 0.9} plots a 90\% CI).
Only applies to \code{type = "emm"} or \code{type = "eff"}.}

\item{p.kr}{logical, if \code{TRUE}, p-value estimation is based on conditional 
F-tests with Kenward-Roger approximation for the df. Caution: Computation
may take very long time for large samples!}

\item{grid.breaks}{numeric; sets the distance between breaks for the axis, 
i.e. at every \code{grid.breaks}'th position a major grid is being printed.}

\item{xlim}{numeric vector of length two, defining lower and upper axis limits
of the x scale. By default, this argument is set to \code{NULL}, i.e. the 
x-axis fits to the required range of the data.}

\item{ylim}{numeric vector of length two, defining lower and upper axis limits
of the y scale. By default, this argument is set to \code{NULL}, i.e. the 
y-axis fits to the required range of the data.}

\item{y.offset}{numeric, offset for text labels when their alignment is adjusted 
to the top/bottom of the geom (see \code{hjust} and \code{vjust}).}

\item{digits}{numeric, amount of digits after decimal point when rounding estimates and values.}

\item{facet.grid}{\code{TRUE} to arrange the lay out of of multiple plots 
in a grid of an integrated single plot.}

\item{prnt.plot}{logical, if \code{TRUE} (default), plots the results as graph. Use \code{FALSE} if you don't
want to plot any graphs. In either case, the ggplot-object will be returned as value.}

\item{...}{other arguments, passed down to the \code{\link[effects]{effect}} resp. 
\code{\link[effects]{allEffects}} function when \code{type = "eff"}.}
}
\value{
(Insisibily) returns the ggplot-objects with the complete plot-list (\code{plot.list})
          as well as the data frames that were used for setting up the ggplot-objects (\code{data.list}).
}
\description{
Plot regression (predicted values) or probability lines (predicted probabilities) of 
               significant interaction terms to better understand effects
               of moderations in regression models. This function accepts following fitted model classes:
               \itemize{
                 \item linear models (\code{\link{lm}})
                 \item generalized linear models (\code{\link{glm}})
                 \item linear mixed effects models (\code{\link[lme4]{lmer}})
                 \item generalized linear mixed effects models (\code{\link[lme4]{glmer}})
                 \item non-linear mixed effects models (\code{\link[lme4]{nlmer}})
                 \item linear mixed effects models (\code{\link[nlme]{lme}}, but only for \code{type = "eff"})
                 \item generalized least squares models (\code{\link[nlme]{gls}}, but only for \code{type = "eff"})
                 \item panel data estimators (\code{\link[plm]{plm}})
               }
               Note that beside interaction terms, also the single predictors of each interaction (main effects)
               must be included in the fitted model as well. Thus, \code{lm(dep ~ pred1 * pred2)} will work, 
               but \code{lm(dep ~ pred1:pred2)} won't!
}
\details{
\describe{
           \item{\code{type = "eff"}}{plots the overall effects (marginal effects) of the interaction, with all remaining
             covariates set to the mean. Effects are calculated using the \code{\link[effects]{effect}}-
             function from the \pkg{effects}-package.
             You can pass further arguments down to \code{allEffects} for flexible
             function call via the \code{...}-argument.
           }
           \item{\code{type = "cond"}}{plots the effective \emph{change} or \emph{impact} 
             (conditional effect) on a dependent variable of a moderation effect, as 
             described by Grace-Martin, i.e. the difference of the moderation effect on the 
             dependent variable in \emph{presence} and \emph{absence} of the moderating effect 
             (\emph{simple slope} plot or \emph{conditional effect}, see Hayes 2012). All
             remaining predictors are set to zero (i.e. ignored and not adjusted for).
             Hence, this plot type may be used especially for \emph{binary or dummy coded} 
             moderator values (see also Esarey and Summer 2015).
             This type \emph{does not} show the overall effect (marginal mean, i.e. adjusted
             for all other predictors and covariates) of interactions on the result of Y. Use 
             \code{type = "eff"} for effect displays similar to the \code{\link[effects]{effect}}-function 
             from the \pkg{effects}-package.
           }
           \item{\code{type = "emm"}}{plots the estimated marginal means of repeated measures designs,
             like two-way repeated measures AN(C)OVA. In detail, this type plots estimated marginal means 
             (also called \emph{least square means} or \emph{marginal means}) of (significant) interaction terms.
             The fitted models may be linear (mixed effects) 
             models of class \code{\link{lm}} or \code{\link[lme4]{merMod}}. This function may be used, for example,
             to plot differences in interventions between control and treatment groups over multiple time points.
           }
         }
         The argument \code{int.term} only applies to \code{type = "eff"} and can be used
         to select a specific interaction term of the model that should be plotted. The function
         then calls \code{effect(int.term, fit)} to compute effects for this specific interaction
         term only. This approach is recommended, when the fitted model contains many observations
         and/or variables, which may slow down the effect-computation dramatically. In such cases,
         consider computing effects for selected interaction terms only with \code{int.terms}.
         See 'Examples'.
}
\note{
Note that beside interaction terms, also the single predictors of each interaction (main effects)
       must be included in the fitted model as well. Thus, \code{lm(dep ~ pred1 * pred2)} will work, 
       but \code{lm(dep ~ pred1:pred2)} won't! \cr \cr
       For \code{type = "emm"}, all interaction terms have to be factors.
       Furthermore, for \code{type = "eff"}, predictors of interactions that are introduced first into the model
       are used as grouping variable, while the latter predictor is printed along the x-axis
       (i.e. lm(y~a+b+a:b) means that "a" is used as grouping variable and "b" is plotted along the x-axis).
}
\examples{
# Note that the data sets used in this example may not be perfectly suitable for
# fitting linear models. I just used them because they are part of the R-software.

# fit "dummy" model. Note that moderator should enter
# first the model, followed by predictor. Else, use
# argument "swap.pred" to change predictor on 
# x-axis with moderator
fit <- lm(weight ~ Diet * Time, data = ChickWeight)

# show summary to see significant interactions
summary(fit)

# plot regression line of interaction terms, including value labels
sjp.int(fit, type = "eff", show.values = TRUE)


# load sample data set
library(sjmisc)
data(efc)
# create data frame with variables that should be included
# in the model
mydf <- data.frame(usage = efc$tot_sc_e,
                   sex = efc$c161sex,
                   education = efc$c172code,
                   burden = efc$neg_c_7,
                   dependency = efc$e42dep)
# convert gender predictor to factor
mydf$sex <- relevel(factor(mydf$sex), ref = "2")
# fit "dummy" model
fit <- lm(usage ~ .*., data = mydf)
summary(fit)

# plot interactions. note that type = "cond" only considers 
# significant interactions by default. use "plevel" to 
# adjust p-level sensivity
sjp.int(fit, type = "cond")

# plot only selected interaction term for
# type = "eff"
sjp.int(fit, type = "eff", int.term = "sex*education")

# plot interactions, using mean and sd as moderator
# values to calculate interaction effect
sjp.int(fit, type = "eff", mdrt.values = "meansd")
sjp.int(fit, type = "cond", mdrt.values = "meansd")

# plot interactions, including those with p-value up to 0.1
sjp.int(fit, type = "cond", plevel = 0.1)

# -------------------------------
# Predictors for negative impact of care.
# Data from the EUROFAMCARE sample dataset
# -------------------------------
library(sjmisc)
data(efc)
# create binary response
y <- ifelse(efc$neg_c_7 < median(stats::na.omit(efc$neg_c_7)), 0, 1)
# create data frame for fitted model
mydf <- data.frame(y = as.factor(y),
                   sex = as.factor(efc$c161sex),
                   barthel = as.numeric(efc$barthtot))
# fit model
fit <- glm(y ~ sex * barthel, data = mydf, family = binomial(link = "logit"))
# plot interaction, increase p-level sensivity
sjp.int(fit, type = "eff", legend.labels = get_labels(efc$c161sex), plevel = 0.1)
sjp.int(fit, type = "cond", legend.labels = get_labels(efc$c161sex), plevel = 0.1)
        
\dontrun{
# -------------------------------
# Plot estimated marginal means
# -------------------------------
# load sample data set
library(sjmisc)
data(efc)
# create data frame with variables that should be included
# in the model
mydf <- data.frame(burden = efc$neg_c_7,
                   sex = efc$c161sex,
                   education = efc$c172code)
# convert gender predictor to factor
mydf$sex <- factor(mydf$sex)
mydf$education <- factor(mydf$education)
# name factor levels and dependent variable
levels(mydf$sex) <- c("female", "male")
levels(mydf$education) <- c("low", "mid", "high")
mydf$burden <- set_label(mydf$burden, "care burden")
# fit "dummy" model
fit <- lm(burden ~ .*., data = mydf)
summary(fit)

# plot marginal means of interactions, no interaction found
sjp.int(fit, type = "emm")
# plot marginal means of interactions, including those with p-value up to 1
sjp.int(fit, type = "emm", plevel = 1)
# swap predictors
sjp.int(fit, type = "emm", plevel = 1, swap.pred = TRUE)

# -------------------------------
# Plot effects
# -------------------------------
# add continuous variable
mydf$barthel <- efc$barthtot
# re-fit model with continuous variable
fit <- lm(burden ~ .*., data = mydf)

# plot effects
sjp.int(fit, type = "eff", show.ci = TRUE)

# plot effects, faceted
sjp.int(fit, type = "eff", int.plot.index = 3, show.ci = TRUE, facet.grid = TRUE)}

}
\references{
\itemize{
             \item Aiken and West (1991). Multiple Regression: Testing and Interpreting Interactions.
             \item Brambor T, Clark WR and Golder M (2006) Understanding Interaction Models: Improving Empirical Analyses. Political Analysis 14: 63-82 \href{https://files.nyu.edu/mrg217/public/pa_final.pdf}{download}
             \item Esarey J, Sumner JL (2015) Marginal Effects in Interaction Models: Determining and Controlling the False Positive Rate. \href{http://jee3.web.rice.edu/interaction-overconfidence.pdf}{download}
             \item Fox J (2003) Effect displays in R for generalised linear models. Journal of Statistical Software 8:15, 1–27, \href{http://www.jstatsoft.org/v08/i15/}{<http://www.jstatsoft.org/v08/i15/>}
             \item Hayes AF (2012) PROCESS: A versatile computational tool for observed variable mediation, moderation, and conditional process modeling [White paper] \href{http://imaging.mrc-cbu.cam.ac.uk/statswiki/FAQ/SobelTest?action=AttachFile&do=get&target=process.pdf}{download}
             \item \href{http://www.theanalysisfactor.com/interpreting-interactions-in-regression/}{Grace-Martin K: Interpreting Interactions in Regression}
             \item \href{http://www.theanalysisfactor.com/clarifications-on-interpreting-interactions-in-regression/}{Grace-Martin K: Clarifications on Interpreting Interactions in Regression}
             }
}
\seealso{
\href{http://www.strengejacke.de/sjPlot/sjp.int/}{sjPlot manual: sjp.int}
}

