#####---------------------------------------------------------------------------
## Grubbs-Patnaik chi^2 distribution
#####---------------------------------------------------------------------------

## determine Grubbs parameters from eigenvalues of covariance matrix
## not vectorized
getGrubbsParam <-
function(sigma, ctr, accuracy=FALSE) {
    if(!is.matrix(sigma))  { stop("sigma must be a matrix") }
    if(!is.numeric(sigma)) { stop("sigma must be numeric") }
    if(missing(ctr))       { ctr <- numeric(ncol(sigma)) }

    eig    <- eigen(sigma)               # eigen-system of covariance matrix
    lambda <- eig$values                 # eigenvalue vector
    eVec   <- eig$vectors                # eigenvector matrix

    if(!all(lambda >= -sqrt(.Machine$double.eps) * abs(lambda[1]))) {
        stop("sigma is numerically not positive definite")
    }

    ## take systematic location bias into account?
    ## if so -> rotate ctr with eigenvectors (decorrelated data)
    ## non-centrality parameters delta = ai^2
    deltaI <- if(accuracy) { (t(eVec) %*% ctr)^2 / lambda } else { 0 }

    ## distribution parameters
    m <- sum(lambda) + sum(lambda*deltaI)    # mean
    v <- 2*sum(lambda^2) + 4*sum(lambda^2*deltaI)  # variance
    n <- 2*m^2 / v                       # df chi-square Patnaik

    ## for Pearson approximation
    mu3 <- 8*sum(lambda^3 + 3*lambda^3*deltaI) # 3rd moment = skewness
    beta1  <- mu3^2 / v^3
    nPrime <- 8/beta1                    # df chi-square Pearson

    ## for Liu, Tang & Zhang approximation
    c1 <- m
    c2 <- v/2
    c3 <- sum(lambda^3) + 3*sum(lambda^3*deltaI)
    c4 <- sum(lambda^4) + 4*sum(lambda^4*deltaI)
    s1 <- c3/sqrt(c2^3)
    s2 <- c4/c2^2
    if(s1^2 > s2) {
        a     <- 1 / (s1-sqrt(s1^2 - s2))
        delta <- s1*a^3 - a^2
        l     <- a^2 - 2*delta
    }  else {
        a     <- 1/s1
        delta <- 0
        l     <- c2^3/c3^2
    }

    muX  <- l + delta
    varX <- 2*a^2

    return(list(m=m, v=v, n=n, nPrime=nPrime, muX=muX, varX=varX, l=l, delta=delta))
}

## determine Grubbs parameters (accuracy=FALSE) from Hoyt parameters
## vectorized
getGPfromHP <-
function(qpar, omega) {
    nnaQ <- which(!is.na(qpar))
    nnaO <- which(!is.na(omega))
    stopifnot(all(qpar[nnaQ] > 0), all(qpar[nnaQ] < 1), all(omega[nnaO] > 0))

    ## eigenvalues from q and omega
    ev2 <- omega / ((1/qpar^2) + 1)
    ev1 <- omega - ev2

    ## distribution parameters
    m <- ev1+ev2                         # mean
    v <- 2*(ev1^2 + ev2^2)               # variance
    n <- 2*m^2 / v                       # df chi-square Patnaik

    ## for Pearson approximation
    mu3    <- 8*(ev1^3 + ev2^3)          # 3rd moment
    beta1  <- mu3^2 / v^3
    nPrime <- 8/beta1                    # df chi-square Pearson

    ## for Liu, Tang & Zhang approximation
    c1 <- m
    c2 <- v/2
    c3 <- ev1^3 + ev2^3
    c4 <- ev1^4 + ev2^4
    s1 <- c3/sqrt(c2^3)
    s2 <- c4/c2^2
    if(s1^2 > s2) {
        a     <- 1 / (s1-sqrt(s1^2 - s2))
        delta <- s1*a^3 - a^2
        l     <- a^2 - 2*delta
    }  else {
        a     <- 1/s1
        delta <- 0
        l     <- c2^3/c3^2
    }

    muX  <- l + delta
    varX <- 2*a^2

    return(list(m=m, v=v, n=n, nPrime=nPrime, muX=muX, varX=varX, l=l, delta=delta))
}

## cdf - vectorized
pChisqGrubbs <-
function(q, m, v, n, nPrime, muX, varX, l, delta, lower.tail=TRUE,
    type=c("Patnaik", "Pearson", "Liu")) {
    type <- match.arg(type)

    pp <- if(type == "Patnaik") {
        qq <- q^2*2*m/v
        pchisq(qq, df=n, lower.tail=lower.tail)
    } else if(type == "Pearson") {
        qq <- (q^2-m)*sqrt(2*nPrime/v) + nPrime
        pchisq(qq, df=nPrime, lower.tail=lower.tail)
    } else if(type == "Liu") {
        qq <- (q^2-m)*sqrt(varX/v) + muX
        pchisq(qq, df=l, ncp=delta, lower.tail=lower.tail)
    }

    return(pp)
}

## quantile function - vectorized
qChisqGrubbs <-
function(p, m, v, n, nPrime, muX, varX, l, delta, lower.tail=TRUE,
    type=c("Patnaik", "Pearson", "Liu")) {
    type <- match.arg(type)

    qq <- if(type == "Patnaik") {
        q1 <- qchisq(p, df=n,      lower.tail=lower.tail)
        sqrt((v/(2*m)) * q1)
    } else if(type == "Pearson") {
        q1 <- qchisq(p, df=nPrime, lower.tail=lower.tail)
        sqrt((q1-nPrime)*sqrt(v/(2*nPrime)) + m)
    } else if(type == "Liu") {
        q1 <- qchisq(p, df=l, ncp=delta, lower.tail=lower.tail)
        sqrt((q1-muX) * sqrt(v/varX) + m)
    }

    return(qq)
}
