% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model-function.R, R/model-in-situ.R,
%   R/model-radar-gun.R, R/model-tether.R, R/model-time-distance.R,
%   R/model-timing-gates.R
\name{model_functions}
\alias{model_functions}
\alias{model_in_situ}
\alias{model_radar_gun}
\alias{model_laser_gun}
\alias{model_tether}
\alias{model_tether_DC}
\alias{model_time_distance}
\alias{model_time_distance_FD}
\alias{model_time_distance_TC}
\alias{model_time_distance_DC}
\alias{model_time_distance_TC_DC}
\alias{model_timing_gates}
\alias{model_timing_gates_TC}
\alias{model_timing_gates_FD}
\alias{model_timing_gates_DC}
\alias{model_timing_gates_TC_DC}
\title{Model functions}
\usage{
model_in_situ(
  velocity,
  acceleration,
  weights = 1,
  velocity_threshold = NULL,
  velocity_step = 0.2,
  n_observations = 2,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_radar_gun(
  time,
  velocity,
  weights = 1,
  CV = NULL,
  use_observed_MSS = FALSE,
  na.rm = FALSE,
  ...
)

model_laser_gun(
  time,
  velocity,
  weights = 1,
  CV = NULL,
  use_observed_MSS = FALSE,
  na.rm = FALSE,
  ...
)

model_tether(
  distance,
  velocity,
  weights = 1,
  CV = NULL,
  use_observed_MSS = FALSE,
  na.rm = FALSE,
  ...
)

model_tether_DC(
  distance,
  velocity,
  weights = 1,
  CV = NULL,
  use_observed_MSS = FALSE,
  na.rm = FALSE,
  ...
)

model_time_distance(time, distance, weights = 1, CV = NULL, na.rm = FALSE, ...)

model_time_distance_FD(
  time,
  distance,
  weights = 1,
  FD = NULL,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_time_distance_TC(
  time,
  distance,
  weights = 1,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_time_distance_DC(
  time,
  distance,
  weights = 1,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_time_distance_TC_DC(
  time,
  distance,
  weights = 1,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_timing_gates(distance, time, weights = 1, CV = NULL, na.rm = FALSE, ...)

model_timing_gates_TC(
  distance,
  time,
  weights = 1,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_timing_gates_FD(
  distance,
  time,
  weights = 1,
  FD = NULL,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_timing_gates_DC(
  distance,
  time,
  weights = 1,
  CV = NULL,
  na.rm = FALSE,
  ...
)

model_timing_gates_TC_DC(
  distance,
  time,
  weights = 1,
  CV = NULL,
  na.rm = FALSE,
  ...
)
}
\arguments{
\item{weights}{Numeric vector. Default is 1}

\item{velocity_threshold}{Velocity cutoff. If \code{NULL} (default), velocity of the observation with
the fastest acceleration is taken as the cutoff value}

\item{velocity_step}{Velocity increment size for finding max acceleration. Default is 0.2 m/s}

\item{n_observations}{Number of top acceleration observations to keep in velocity bracket.
Default is 2}

\item{CV}{Should cross-validation be used to estimate model fit? Default is \code{NULL}. Otherwise use integer
indicating number of folds}

\item{na.rm}{Logical. Default is FALSE}

\item{...}{Forwarded to \code{\link[minpack.lm]{nlsLM}} function}

\item{time, velocity, distance, acceleration}{Numeric vector}

\item{use_observed_MSS}{Should observed peak \code{velocity} be used as \code{MSS} parameter? Default
is \code{FALSE}}

\item{FD}{Use this parameter if you do not want the \code{FD} parameter to be estimated, but rather take the
provided value}
}
\value{
List object with the following elements:
    \describe{
        \item{data}{Data frame used to estimate the sprint parameters}
        \item{model_info}{Extra information regarding model used}
        \item{model}{Model returned by the \code{\link[minpack.lm]{nlsLM}} function}
        \item{parameters}{List with the following estimated parameters:
            \code{MSS}, \code{MAC}, \code{TAU}, and \code{PMAX}}
        \item{correction}{List with additional model correcitons}
        \item{predictions}{Data frame with \code{.predictor}, \code{.observed},
            \code{.predicted}, and \code{.residual} columns}
        \item{model_fit}{List with multiple model fit estimators}
        \item{CV}{If cross-validation is performed, this will included the data as above, but
             for each fold}
        }
}
\description{
Family of functions that serve a purpose of estimating short sprint parameters

\code{model_in_situ} estimates short sprint parameters using velocity-acceleration trace,
    provided by the monitoring systems like GPS or LPS. See references for the information

\code{model_radar_gun} estimates short sprint parameters using time-velocity trace,
    with additional parameter \code{TC} serving as intercept

\code{model_laser_gun} alias for \code{\link{model_radar_gun}}

\code{model_tether} estimates short sprint parameters using distance-velocity trace
    (e.g., tether devices).

\code{model_tether_DC} estimates short sprint parameters using distance-velocity trace
    (e.g., tether devices) with additional distance correction \code{DC} parameter

\code{model_time_distance} estimates short sprint parameters using time distance trace

\code{model_time_distance_FD} estimates short sprint parameters using time-distance trace
     with additional flying distance correction parameter \code{FD}

\code{model_time_distance} estimates short sprint parameters using time distance trace
    with additional time correction parameter \code{TC}

\code{model_time_distance} estimates short sprint parameters using time distance trace
    with additional distance correction parameter \code{DC}

\code{model_time_distance} estimates short sprint parameters using time distance trace
    with additional time correction \code{TC} and distance correction \code{TC} parameters

\code{model_timing_gates} estimates short sprint parameters using distance-time trace
     (e.g., timing gates/photo cells)

\code{model_timing_gates_TC} estimates short sprint parameters using distance-time trace
     (e.g., timing gates/photo cells), with additional time correction parameter \code{TC}

\code{model_timing_gates_FD} estimates short sprint parameters using distance-time trace
     (e.g., timing gates/photo cells), with additional flying distance correction parameter \code{FD}

\code{model_timing_gates_DC} estimates short sprint parameters using distance-time trace
     (e.g., timing gates/photo cells), with additional distance correction parameter \code{DC}

\code{model_timing_gates_TC_DC} estimates short sprint parameters using distance-time trace
     (e.g., timing gates/photo cells), with additional time correction \code{TC} and
     distance correction \code{DC} parameters
}
\examples{

# Model In-Situ (Embedded profiling)
data("LPS_session")
m1 <- model_in_situ(
  velocity = LPS_session$velocity,
  acceleration = LPS_session$acceleration,
  # Use specific cutoff value
  velocity_threshold = 4)
m1
plot(m1)


# Model Radar Gun (includes Time Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 6, 0.1))

# Add some noise
df$velocity <- df$velocity + rnorm(n = nrow(df), 0, 10^-2)

m1 <- model_radar_gun(time = df$time, velocity = df$velocity)
m1
plot(m1)


# Model Laser Gun (includes Time Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 6, 0.1))

# Add some noise
df$velocity <- df$velocity + rnorm(n = nrow(df), 0, 10^-2)

m1 <- model_laser_gun(time = df$time, velocity = df$velocity)
m1
plot(m1)


# Model Tether
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 6, 0.5))
m1 <- model_tether(distance = df$distance, velocity = df$velocity)
m1
plot(m1)


# Model Tether with Distance Correction (DC)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0.001, 6, 0.5), DC = 5)
m1 <- model_tether_DC(distance = df$distance, velocity = df$velocity)
m1
plot(m1)


# Model Time-Distance trace (simple, without corrections)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 5, by = 0.5))
m1 <- model_time_distance(time = df$time, distance = df$distance)
m1
plot(m1)


# Model Time-Distance trace (with Flying Distance Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 5, by = 0.5), FD = 0.5)
m1 <- model_time_distance_FD(time = df$time, distance = df$distance)
m1
plot(m1)


# Model Time-Distance trace (with Time Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 5, by = 0.5), TC = 1.5)
m1 <- model_time_distance_TC(time = df$time, distance = df$distance)
m1
plot(m1)


# Model Time-Distance trace (with Distance Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 5, by = 0.5), DC = -5)
m1 <- model_time_distance_DC(time = df$time, distance = df$distance)
m1
plot(m1)


# Model Time-Distance trace (with Time and Distance Corrections)
df <- create_sprint_trace(MSS = 8, MAC = 6, time = seq(0, 5, by = 0.5), TC = -1.3, DC = 5)
m1 <- model_time_distance_TC_DC(time = df$time, distance = df$distance)
m1
plot(m1)


# Model Timing Gates (simple, without corrections)
df <- create_sprint_trace(MSS = 8, MAC = 6, distance = c(5, 10, 20, 30, 40))
m1 <- model_timing_gates(distance = df$distance, time = df$time)
m1
plot(m1)


# Model Timing Gates (with Time Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, distance = c(5, 10, 20, 30, 40), TC = 0.2)
m1 <- model_timing_gates_TC(distance = df$distance, time = df$time)
m1
plot(m1)


# Model Timing Gates (with Flying Distance Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, distance = c(5, 10, 20, 30, 40), FD = 0.5)
m1 <- model_timing_gates_FD(distance = df$distance, time = df$time)
m1
plot(m1)


# Model Timing Gates (with Distance Correction)
df <- create_sprint_trace(MSS = 8, MAC = 6, distance = c(5, 10, 20, 30, 40), DC = 1.5)
m1 <- model_timing_gates_DC(distance = df$distance, time = df$time)
m1
plot(m1)


# Model Timing Gates (with Time and Distance Corrections)
df <- create_sprint_trace(MSS = 8, MAC = 6, distance = c(5, 10, 20, 30, 40), TC = 0.25, DC = 1.5)
m1 <- model_timing_gates_TC_DC(distance = df$distance, time = df$time)
m1
plot(m1)

}
\references{
Samozino P. 2018. A Simple Method for Measuring Force, Velocity and Power Capabilities and Mechanical
        Effectiveness During Sprint Running. In: Morin J-B, Samozino P eds. Biomechanics of Training and Testing.
       Cham: Springer International Publishing, 237–267. DOI: 10.1007/978-3-319-05633-3_11.

Clavel, P., Leduc, C., Morin, J.-B., Buchheit, M., & Lacome, M. (2023).
    Reliability of individual acceleration-speed profile in-situ in elite youth
    soccer players. Journal of Biomechanics, 153, 111602.
    https://doi.org/10.1016/j.jbiomech.2023.111602

    Morin, J.-B. (2021). The “in-situ” acceleration-speed profile for team
    sports: testing players without testing them. JB Morin, PhD – Sport Science website.
    Accessed 31. Dec. 2023.
    https://jbmorin.net/2021/07/29/the-in-situ-sprint-profile-for-team-sports-testing-players-without-testing-them/
}
