% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SoilTaxonomyDendrogram.R
\name{SoilTaxonomyDendrogram}
\alias{SoilTaxonomyDendrogram}
\title{Soil Taxonomy Dendrogram}
\usage{
SoilTaxonomyDendrogram(
  spc,
  name = "hzname",
  name.style = "right-center",
  rotationOrder = NULL,
  max.depth = 150,
  n.depth.ticks = 6,
  scaling.factor = 0.015,
  cex.names = 0.75,
  cex.id = 0.75,
  axis.line.offset = -4,
  width = 0.1,
  y.offset = 0.5,
  shrink = FALSE,
  font.id = 2,
  cex.taxon.labels = 0.66,
  dend.color = par("fg"),
  dend.width = 1,
  ...
)
}
\arguments{
\item{spc}{a \code{SoilProfileCollection} object, typically returned by \code{soilDB::fetchOSD}}

\item{name}{column name containing horizon names}

\item{name.style}{passed to \code{aqp::plotSPC} (default: "right-center")}

\item{rotationOrder}{numeric vector with desired ordering of leaves in the dendrogram from left to right, or character vector matching profile IDs}

\item{max.depth}{depth at which profiles are truncated for plotting}

\item{n.depth.ticks}{suggested number of ticks on the depth axis}

\item{scaling.factor}{scaling factor used to convert depth units into plotting units}

\item{cex.names}{character scaling for horizon names}

\item{cex.id}{character scaling for profile IDs}

\item{axis.line.offset}{horizontal offset for depth axis}

\item{width}{width of profiles}

\item{y.offset}{vertical offset between dendrogram and profiles}

\item{shrink}{logical, should long horizon names be shrunk by 80\% ?}

\item{font.id}{font style applied to profile id, default is 2 (bold)}

\item{cex.taxon.labels}{character scaling for taxonomic information}

\item{dend.color}{dendrogram line color}

\item{dend.width}{dendrogram line width}

\item{...}{additional arguments to \code{aqp::plotSPC}}
}
\value{
An invisibly-returned list containing:
\itemize{
\item \code{dist}: pair-wise dissimilarity matrix
\item \code{order}: final ordering of hclust leaves
}
}
\description{
Plot a dendrogram based on the first 4 levels of Soil Taxonomy, with soil profiles hanging below. A dissimilarity matrix is computed using Gower's distance metric for nominal-scale variables, based on order, sub order, great group, and subgroup level taxa. See the Details and Examples sections below for more information.
}
\details{
This function looks for specific site-level attributes named: \code{soilorder}, \code{suborder}, \code{greatgroup}, and \code{subgroup}.

The \code{rotationOrder} argument uses (requires) the \code{dendextend::rotate()} function to re-order leaves within the \code{hclust} representation of the ST hierarchy. Perfect sorting is not always possible.
}
\examples{

\donttest{

if(requireNamespace("curl") &
   curl::has_internet() &
   require(aqp) &
   require(soilDB)
) {
  
  
  
  # soils of interest
  s.list <- c('musick', 'cecil', 'drummer', 'amador', 'pentz', 'reiff', 
              'san joaquin','montpellier','grangeville','pollasky','ramona')
  
  # fetch and convert data into an SPC
  h <- fetchOSD(s.list)
  
  # plot dendrogram + profiles
  SoilTaxonomyDendrogram(h)
  
  # again, this time save the pair-wise dissimilarity matrix
  # note that there isn't a lot of discrimination between soils
  (d <- SoilTaxonomyDendrogram(h))
  
  
  # a different set
  soils <- c('cecil', 'altavista', 'lloyd', 'wickham', 'wilkes',
             'chewacla', 'congaree')
  
  # get morphology + extended summaries for sorting of dendrogram
  s <- fetchOSD(soils, extended = TRUE)
  
  # get summary and ignore the figure
  res <- vizHillslopePosition(s$hillpos)
  
  # compare default sorting to soils sorting according to catenary, e.g.
  # hillslope position
  par(mar=c(0,0,1,1), mfrow=c(2,1))
  
  SoilTaxonomyDendrogram(s$SPC, width=0.25)
  mtext('default sorting', side = 2, line=-1, font=3, cex=1.25)
  
  SoilTaxonomyDendrogram(s$SPC, rotationOrder = res$order, width=0.25)
  mtext('approx. catenary sorting', side = 2, line=-1, font=3, cex=1.25)
 
  
# classic chronosequence from the San Joaquin Valley, CA
library(aqp)
library(soilDB)
library(sharpshootR)

s <- c('tujunga', 'hanford', 'greenfield', 'snelling', 'san joaquin')
osds <- fetchOSD(s)

idx <- match(toupper(s), profile_id(osds))

# encode horizon boundarydistinctness via vertical offset
osds$hd <- hzDistinctnessCodeToOffset(
  osds$distinctness, 
  codes=c('very abrupt', 'abrupt', 'clear', 'gradual', 'diffuse')
)

# encode horizon boundary topography via vertical offset
osds$hzto <- hzTopographyCodeToOffset(
  osds$topography, 
  codes = c('smooth', 'wavy', 'irregular', 'broken')
)

# also encode horizon boundary topography las line type
osds$hzto.lty <- hzTopographyCodeToLineType(
  osds$topography,
  codes = c('smooth', 'wavy', 'irregular', 'broken')
)

# label data source, used later 
site(osds)$source <- 'OSD'

# concise representation of hz bnd distinctness and topography
# similar to field notes
osds$bnd.code <- sprintf(
  "\%s\%s",
  substr(osds$distinctness, 1, 1),
  substr(osds$topography, 1, 1)
)

# remove NA
osds$bnd.code <- gsub('NANA', '', osds$bnd.code)

par(mar = c(0, 0, 0, 1), bg = 'black', fg = 'white')

plotSPC(
osds, 
plot.order = idx, 
width = 0.3, 
name.style = 'center-center', 
cex.names = 0.66, 
plot.depth.axis = FALSE, 
hz.depths = TRUE, 
shrink = TRUE, 
hz.distinctness.offset = 'hd', 
hz.topography.offset = 'hzto', 
hz.boundary.lty = 'hzto.lty'
)

legend(
'bottomright', 
horiz = TRUE, 
legend = c('Smooth', 'Wavy', 'Irregular', 'Broken'), 
lty = 1:4, 
inset = 0.05, 
bty = 'n', 
cex = 0.85
)

# note that `rotationOrder` uses the ordering of series names (uppercase to match profile IDs)
# to re-order the terminal branches of the dendrogram
SoilTaxonomyDendrogram(
osds, 
rotationOrder = toupper(s), 
cex.taxon.labels = 0.85, 
width = 0.3, 
name.style = 'center-center', 
cex.names = 0.66, 
plot.depth.axis = FALSE, 
hz.depths = TRUE, 
shrink = TRUE, 
hz.distinctness.offset = 'hd', 
hz.topography.offset = 'hzto', 
hz.boundary.lty = 'hzto.lty'
)

legend(
'bottomright', 
horiz = TRUE, 
legend = c('Smooth', 'Wavy', 'Irregular', 'Broken'), 
lty = 1:4, 
inset = 0.05, 
bty = 'n', 
cex = 0.85
)
     
}

}

}
\author{
D.E. Beaudette
}
