% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitSpectraSMC.R
\name{fitSpectraSMC}
\alias{fitSpectraSMC}
\title{Fit the model using Sequential Monte Carlo (SMC).}
\usage{
fitSpectraSMC(wl, spc, peakWL, lPriors, conc = rep(1, nrow(spc)),
  npart = 10000, rate = 0.9, minESS = npart/2, destDir = NA)
}
\arguments{
\item{wl}{Vector of \code{nwl} wavenumbers at which the spetra are observed.}

\item{spc}{\code{n_y * nwl} Matrix of observed Raman spectra.}

\item{peakWL}{Vector of locations for each peak (cm^-1)}

\item{lPriors}{List of hyperparameters for the prior distributions.}

\item{conc}{Vector of \code{n_y} nanomolar (nM) dye concentrations for each observation.}

\item{npart}{number of SMC particles to use for the importance sampling distribution.}

\item{rate}{the target rate of reduction in the effective sample size (ESS).}

\item{minESS}{minimum effective sample size, below which the particles are resampled.}

\item{destDir}{destination directory to save intermediate results (for long-running computations)}
}
\value{
a List containing weighted parameter values, known as particles:
\describe{
  \item{\code{weights}}{Vector of importance weights for each particle.}
  \item{\code{beta}}{\code{npart * npeaks} Matrix of regression coefficients for the amplitudes.}
  \item{\code{scale}}{\code{npart * npeaks} Matrix of scale parameters.}
  \item{\code{sigma}}{Vector of \code{npart} standard deviations.}
  \item{\code{alpha}}{\code{bl.knots * n_y * npart} Array of spline coefficients for the baseline.}
  \item{\code{basis}}{A dense \code{nwl * bl.knots} Matrix containing the values of the basis functions.}
  \item{\code{expFn}}{\code{npart * nwl} Matrix containing the spectral signature.}
  \item{\code{ess}}{Vector containing the effective sample size (ESS) at each SMC iteration.}
  \item{\code{logEvidence}}{Vector containing the logarithm of the model evidence (marginal likelihood).}
  \item{\code{accept}}{Vector containing the Metropolis-Hastings acceptance rate at each SMC iteration.}
  \item{\code{sd.mh}}{\code{niter * 2npeaks} Matrix of random walk MH bandwidths at each SMC iteration..}
  }
}
\description{
Fit the model using Sequential Monte Carlo (SMC).
}
\examples{
wavenumbers <- seq(200,600,by=10)
spectra <- matrix(nrow=1, ncol=length(wavenumbers))
peakLocations <- c(300,500)
peakAmplitude <- c(10000,4000)
peakScale <- c(10, 15)
signature <- weightedLorentzian(peakLocations, peakScale, peakAmplitude, wavenumbers)
baseline <- 1000*cos(wavenumbers/200) + 2*wavenumbers
spectra[1,] <- signature + baseline + rnorm(length(wavenumbers),0,200)
lPriors <- list(scale.mu=log(11.6) - (0.4^2)/2, scale.sd=0.4, bl.smooth=10^11, bl.knots=20,
                beta.mu=5000, beta.sd=5000, noise.sd=200, noise.nu=4)
result <- fitSpectraSMC(wavenumbers, spectra, peakLocations, lPriors, npart=500)
}
\references{
Chopin (2002) "A Sequential Particle Filter Method for Static Models," Biometrika 89(3): 539--551,
DOI: \href{http://dx.doi.org/10.1093/biomet/89.3.539}{10.1093/biomet/89.3.539}
}
