% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/poisthin.R
\name{poisthin}
\alias{poisthin}
\title{Apply Poisson thinning to a matrix of count data.}
\usage{
poisthin(
  mat,
  nsamp = nrow(mat),
  ngene = ncol(mat),
  gselect = c("max", "random", "rand_max", "custom", "mean_max"),
  gvec = NULL,
  skip_gene = 0L,
  signal_fun = stats::rnorm,
  signal_params = list(mean = 0, sd = 1),
  prop_null = 1,
  alpha = 0,
  group_assign = c("frac", "random", "cor"),
  group_prop = 0.5,
  corvec = NULL
)
}
\arguments{
\item{mat}{A matrix of count data. The rows index the individuals and
the columns index the genes.}

\item{nsamp}{The number of samples to select from \code{mat}.}

\item{ngene}{The number of genes to select from \code{mat}.}

\item{gselect}{How should we select the subset of genes? Should we choose
the \code{ngene} most median expressed genes (\code{"max"}), a random sample
of the genes (\code{"random"}), a random sample of the most expressed
genes (\code{"rand_max"}), a user-provided list (\code{"custom"}), or by maximum
mean expression level (\code{"mean_max"})?
If \code{"custom"}, then \code{gvec} should be specified. Expression levels
of a gene are measured by median expression across individuals with ties broken
by mean expression.}

\item{gvec}{A logical of length \code{ncol(mat)}. A \code{TRUE} in position \eqn{i}
indicates inclusion into the smaller dataset. Hence, \code{sum(gvec)} should
equal \code{ngene}.}

\item{skip_gene}{The number of maximally expressed genes to skip.
Not used if \code{gselect = "custom"}.}

\item{signal_fun}{A function that returns the signal. This should take as
input \code{n} for the number of samples to return and then return only
a vector of samples.}

\item{signal_params}{A list of additional arguments to pass to \code{signal_fun}.}

\item{prop_null}{The proportion of genes that are null.}

\item{alpha}{If \eqn{b} is an effect and \eqn{s} is an empirical standard deviation, then
we model \eqn{b/s^\alpha} as being exchangeable.}

\item{group_assign}{How should we assign groups? Exactly specifying the
proportion of individuals in each group (\code{"frac"}), with a
Bernoulli distribution (\code{"random"}), or correlated with latent factors
(\code{"cor"})? If \code{group_assign = "cor"}, then you have to specify
\code{corvec}. If \code{group_assign = "frac"} or
\code{group_assign = "random"}, then the proportion of samples in each
group is specified with the \code{group_prop} argument.}

\item{group_prop}{The proportion of individuals that are in group 1.
This proportion is deterministic if \code{group_assign = "frac"}, and
is the expected proportion if \code{group_assign = "random"}. This
argument is not used if \code{group_assign = "cor"}.}

\item{corvec}{A vector of correlations. \code{corvec[i]} is the correlation
of the latent group assignment vector with the ith latent confounder.
Only used if \code{group_assign = "cor"}. This vector is constrained
so that \code{crossprod(corvec) < 1}. The number of latent factors
is taken to be the length of corvec. Note that the correlations of the
latent factors with the observed group-assignment vector (instead of the
latent group-assignment vector) will be \code{corvec * sqrt(2 / pi)}.}
}
\value{
A list with the following elements:
\itemize{
 \item{\code{Y}: }{A matrix of altered counts with \code{nsamp} rows
       and \code{ngene} columns.}
 \item{\code{X}: }{A design matrix. The first column contains a vector ones (for an
       intercept term) and the second column contains an indicator for group membership.}
 \item{\code{beta}: }{The approximately true effect sizes of \eqn{log(Y) ~ X\beta}.}
 \item{\code{corassign}: }{The output from the call to \code{\link{corassign}}.
       Only returned if \code{group_assign = "cor"}.}
}
}
\description{
This is now defunct. Please try out \code{\link{select_counts}} and
\code{\link{thin_2group}}.
}
\details{
Given a matrix of RNA-seq counts, this function will randomly select two groups of
samples and add signal to a known proportion of the genes. This signal
is the log (base 2) effect size of the group indicator in a linear model.
The user may specify the distribution of the effects.

The Poisson thinning approach first randomly assigns samples to be in one of two groups. Then,
given this assignment, will Binomially sample counts with a sample size of the gene expression
counts and a probability that is a function of the effect size. For details, see
Gerard and Stephens (2017).
}
\examples{
## Simulate data from given matrix of counts
## In practice, you would obtain Y from a real dataset, not simulate it.
set.seed(1)
nsamp <- 10
ngene <- 1000
Y <- matrix(stats::rpois(nsamp * ngene, lambda = 50), nrow = ngene)


## Apply thinning
poisout <- poisthin(mat           = t(Y),
                    nsamp         = 9,
                    ngene         = 999,
                    signal_fun    = stats::rnorm,
                    signal_params = list(mean = 0, sd = 1),
                    prop_null     = 0.9)

## Dimension of count matrix is smaller.
dim(poisout$Y)

## Can verify signal was added by estimating it with lm().
betahat <- coef(lm(log2(poisout$Y + 1) ~ poisout$X[, 2]))[2, ]
plot(poisout$beta, betahat, xlab = "Coefficients", ylab = "Estimates")
abline(0, 1, col = 2, lty = 2)

}
\references{
Gerard, D., & Stephens, M. (2017).
    Unifying and generalizing methods for removing unwanted variation based
    on negative controls. arXiv preprint arXiv:1705.08393.
}
\author{
David Gerard
}
