% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_hmm.R
\name{build_hmm}
\alias{build_hmm}
\title{Build a Hidden Markov Model}
\usage{
build_hmm(observations, transition_probs, emission_probs, initial_probs,
  state_names = NULL, channel_names = NULL)
}
\arguments{
\item{observations}{TraMineR stslist (see \code{\link[TraMineR]{seqdef}}) containing
the sequences, or a list of such objects (one for each channel).}

\item{transition_probs}{A matrix of transition probabilities.}

\item{emission_probs}{A matrix of emission probabilities or a list of such
objects (one for each channel). Emission probabilities should follow the
ordering of the alphabet of observations (\code{alphabet(observations)}, returned as \code{symbol_names}).}

\item{initial_probs}{A vector of initial state probabilities.}

\item{state_names}{A list of optional labels for the hidden states. If \code{NULL},
the state names are taken from the row names of the transition matrix. If this is
also \code{NULL}, numbered states are used.}

\item{channel_names}{A vector of optional names for the channels.}
}
\value{
Object of class \code{hmm} with following elements:
\describe{
   \item{\code{observations}}{State sequence object or a list of such containing the data.}
   \item{\code{transition_probs}}{A matrix of transition probabilities.}
   \item{\code{emission_probs}}{A matrix or a list of matrices of emission probabilities.}
   \item{\code{initial_probs}}{A vector of initial probabilities.}
   \item{\code{state_names}}{Names for hidden states.}
   \item{\code{symbol_names}}{Names for observed states.}
   \item{\code{channel_names}}{Names for channels of sequence data.}
   \item{\code{length_of_sequences}}{(Maximum) length of sequences.}
   \item{\code{n_sequences}}{Number of sequences.}
   \item{\code{n_symbols}}{Number of observed states (in each channel).}
   \item{\code{n_states}}{Number of hidden states.}
   \item{\code{n_channels}}{Number of channels.}
}
}
\description{
Function \code{build_hmm} constructs a hidden Markov model object of class \code{hmm}.
}
\details{
Returned model contains some attributes such as \code{nobs} and \code{df}
which define the number of observations in the  model and number of estimable
model parameters, which are used in computing BIC.
When computing \code{nobs} for multichannel model, each observed value in
single channel amounts to $1/C$ observation, i.e. fully observed
time point for single sequences amounts to one observation. For degrees of
freedom \code{df}, zero probabilities of initial model are defined as structural zeroes.
}
\examples{

# Single-channel data

data("mvad", package = "TraMineR")

mvad.alphabet <- c("employment", "FE", "HE", "joblessness", "school",
                   "training")
mvad.labels <- c("employment", "further education", "higher education",
                 "joblessness", "school", "training")
mvad.scodes <- c("EM", "FE", "HE", "JL", "SC", "TR")
mvad.seq <- seqdef(mvad, 17:86, alphabet = mvad.alphabet, states = mvad.scodes,
                   labels = mvad.labels, xtstep = 6)

# Starting values for the emission matrix
emiss <- matrix(NA, nrow = 4, ncol = 6)
emiss[1,] <- seqstatf(mvad.seq[, 1:12])[, 2] + 1
emiss[2,] <- seqstatf(mvad.seq[, 13:24])[, 2] + 1
emiss[3,] <- seqstatf(mvad.seq[, 25:48])[, 2] + 1
emiss[4,] <- seqstatf(mvad.seq[, 49:70])[, 2] + 1
emiss <- emiss / rowSums(emiss)

# Starting values for the transition matrix

tr <- matrix(
  c(0.80, 0.10, 0.05, 0.05,
    0.05, 0.80, 0.10, 0.05,
    0.05, 0.05, 0.80, 0.10,
    0.05, 0.05, 0.10, 0.80),
  nrow=4, ncol=4, byrow=TRUE)

# Starting values for initial state probabilities
init <- c(0.3, 0.3, 0.2, 0.2)

# Building a hidden Markov model with starting values
init_hmm_mvad <- build_hmm(
  observations = mvad.seq, transition_probs = tr,
  emission_probs = emiss, initial_probs = init
)

#########################################


# Multichannel data

# Three-state three-channel hidden Markov model
# See ?hmm_biofam for five-state version

data("biofam3c")

# Building sequence objects
marr.seq <- seqdef(biofam3c$married, start = 15,
  alphabet = c("single", "married", "divorced"))
child.seq <- seqdef(biofam3c$children, start = 15,
  alphabet = c("childless", "children"))
left.seq <- seqdef(biofam3c$left, start = 15,
  alphabet = c("with parents", "left home"))

# Define colors
attr(marr.seq, "cpal") <- c("violetred2", "darkgoldenrod2", "darkmagenta")
attr(child.seq, "cpal") <- c("darkseagreen1", "coral3")
attr(left.seq, "cpal") <- c("lightblue", "red3")

# Starting values for emission matrices

emiss_marr <- matrix(NA, nrow = 3, ncol = 3)
emiss_marr[1,] <- seqstatf(marr.seq[, 1:5])[, 2] + 1
emiss_marr[2,] <- seqstatf(marr.seq[, 6:10])[, 2] + 1
emiss_marr[3,] <- seqstatf(marr.seq[, 11:16])[, 2] + 1
emiss_marr <- emiss_marr / rowSums(emiss_marr)

emiss_child <- matrix(NA, nrow = 3, ncol = 2)
emiss_child[1,] <- seqstatf(child.seq[, 1:5])[, 2] + 1
emiss_child[2,] <- seqstatf(child.seq[, 6:10])[, 2] + 1
emiss_child[3,] <- seqstatf(child.seq[, 11:16])[, 2] + 1
emiss_child <- emiss_child / rowSums(emiss_child)

emiss_left <- matrix(NA, nrow = 3, ncol = 2)
emiss_left[1,] <- seqstatf(left.seq[, 1:5])[, 2] + 1
emiss_left[2,] <- seqstatf(left.seq[, 6:10])[, 2] + 1
emiss_left[3,] <- seqstatf(left.seq[, 11:16])[, 2] + 1
emiss_left <- emiss_left / rowSums(emiss_left)

# Starting values for transition matrix
trans <- matrix(c(0.9, 0.07, 0.03,
                0,  0.9,  0.1,
                0,    0,    1), nrow = 3, ncol = 3, byrow = TRUE)

# Starting values for initial state probabilities
inits <- c(0.9, 0.09, 0.01)

# Building hidden Markov model with initial parameter values
init_hmm_bf <- build_hmm(
  observations = list(marr.seq, child.seq, left.seq),
  transition_probs = trans,
  emission_probs = list(emiss_marr, emiss_child, emiss_left),
  initial_probs = inits)

}
\seealso{
\code{\link{fit_model}} for estimating model parameters.
}

