% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helperFunctions.R
\name{semPower.genSigma}
\alias{semPower.genSigma}
\title{semPower.genSigma}
\usage{
semPower.genSigma(
  Lambda = NULL,
  Phi = NULL,
  Beta = NULL,
  Psi = NULL,
  Theta = NULL,
  tau = NULL,
  Alpha = NULL,
  ...
)
}
\arguments{
\item{Lambda}{factor loading matrix. A list for multiple group models. Can also be specified using various shortcuts, see \code{\link[=genLambda]{genLambda()}}.}

\item{Phi}{for CFA models, factor correlation (or covariance) matrix or single number giving the correlation between all factors or \code{NULL} for uncorrelated factors. A list for multiple group models.}

\item{Beta}{for SEM models, matrix of regression slopes between latent variables (all-y notation). A list for multiple group models.}

\item{Psi}{for SEM models, variance-covariance matrix of latent residuals when \code{Beta} is specified. If \code{NULL}, a diagonal matrix is assumed. A list for multiple group models.}

\item{Theta}{variance-covariance matrix between manifest residuals. If \code{NULL} and \code{Lambda} is not a square matrix, \code{Theta} is diagonal so that the manifest variances are 1. If \code{NULL} and \code{Lambda} is square, \code{Theta} is 0. A list for multiple group models.}

\item{tau}{vector of intercepts. If \code{NULL} and \code{Alpha} is set, these are assumed to be zero. If both \code{Alpha} and \code{tau} are \code{NULL}, no means are returned. A list for multiple group models.}

\item{Alpha}{vector of factor means. If \code{NULL} and \code{tau} is set, these are assumed to be zero. If both \code{Alpha} and \code{tau} are \code{NULL}, no means are returned. A list for multiple group models.}

\item{...}{other}
}
\value{
Returns a list (or list of lists for multiple group models) containing the following components:
\item{\code{Sigma}}{implied variance-covariance matrix.}
\item{\code{mu}}{implied means}
\item{\code{Lambda}}{loading matrix}
\item{\code{Phi}}{covariance matrix of latent variables}
\item{\code{Beta}}{matrix of regression slopes}
\item{\code{Psi}}{residual covariance matrix of latent variables}
\item{\code{Theta}}{residual covariance matrix of observed variables}
\item{\code{tau}}{intercepts}
\item{\code{Alpha}}{factor means}
\item{\code{modelPop}}{\code{lavaan} model string defining the population model}
\item{\code{modelTrue}}{\code{lavaan} model string defining the "true" analysis model freely estimating all non-zero parameters.}
\item{\code{modelTrueCFA}}{\code{lavaan} model string defining a model similar to \code{modelTrue}, but purely CFA based and thus omitting any regression relationships.}
}
\description{
Generate a covariance matrix (and a mean vector) and associated \code{lavaan} model strings based on CFA or SEM model matrices.
}
\details{
This function generates the variance-covariance matrix of the \eqn{p} observed variables \eqn{\Sigma} and their means \eqn{\mu} via a confirmatory factor (CFA) model or a more general structural equation model.

In the CFA model,
\deqn{\Sigma = \Lambda \Phi \Lambda' + \Theta}
where \eqn{\Lambda} is the \eqn{p \cdot m} loading matrix, \eqn{\Phi} is the variance-covariance matrix of the \eqn{m} factors, and \eqn{\Theta} is the residual variance-covariance matrix of the observed variables. The means are
\deqn{\mu = \tau + \Lambda \alpha}
with the \eqn{p} indicator intercepts \eqn{\tau} and the \eqn{m} factor means \eqn{\alpha}.

In the structural equation model,
\deqn{\Sigma = \Lambda (I - B)^{-1} \Psi [(I - B)^{-1}]'  \Lambda' + \Theta }
where \eqn{B} is the \eqn{m \cdot m} matrix containing the regression slopes and \eqn{\Psi} is the (residual) variance-covariance matrix of the \eqn{m} factors. The means are
\deqn{\mu = \tau + \Lambda (I - B)^{-1} \alpha}

In either model, the meanstructure can be omitted, leading to factors with zero means and zero intercepts.

When \eqn{\Lambda = I}, the models above do not contain any factors and reduce to ordinary regression or path models.

If \code{Phi} is defined, a CFA model is used, if \code{Beta} is defined, a structural equation model.
When both \code{Phi} and \code{Beta} are \code{NULL}, a CFA model is used with \eqn{\Phi = I}, i. e., uncorrelated factors.
When \code{Phi} is a single number, all factor correlations are equal to this number.

When \code{Beta} is defined and \code{Psi} is \code{NULL}, \eqn{\Psi = I}.

When \code{Theta} is \code{NULL}, \eqn{\Theta} is a diagonal matrix with all elements such that the variances of the observed variables are 1. When there is only a single observed indicator for a factor, the corresponding element in \eqn{\Theta} is set to zero.

Instead of providing the loading matrix \eqn{\Lambda} via \code{Lambda}, there are several shortcuts (see \code{\link[=genLambda]{genLambda()}}):
\itemize{
\item \code{loadings}: defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading in the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: used in conjunction with \code{loadM} or \code{loadMinmax}, defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: defines the mean loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the mean loadings of the first factor to equal .5 and those of the second factor do equal .6
\item \code{loadSD}: used in conjunction with \code{loadM}, defines the standard deviations of the loadings. If omitted or NULL, the standard deviations are zero. Otherwise, the loadings are sampled from a normal distribution with N(loadM, loadSD) for each factor.
\item \code{loadMinMax}: defines the minimum and maximum loading either for all factors or separately for each factor (as a list). The loadings are then sampled from a uniform distribution. For example, \code{loadMinMax = list(c(.4, .6), c(.4, .8))} defines the loadings for the first factor lying between .4 and .6, and those for the second factor between .4 and .8.
}
}
\examples{
\dontrun{
# single factor model with five indicators each loading by .5
gen <- semPower.genSigma(nIndicator = 5, loadM = .5)
gen$Sigma     # implied covariance matrix
gen$modelTrue # analysis model string
gen$modelPop  # population model string

# orthogonal two factor model with four and five indicators each loading by .5
gen <- semPower.genSigma(nIndicator = c(4, 5), loadM = .5)

# correlated (r = .25) two factor model with 
# four indicators loading by .7 on the first factor 
# and five indicators loading by .6 on the second factor
gen <- semPower.genSigma(Phi = .25, nIndicator = c(4, 5), loadM = c(.7, .6))

# correlated three factor model with variying indicators and loadings, 
# factor correlations according to Phi
Phi <- matrix(c(
  c(1.0, 0.2, 0.5),
  c(0.2, 1.0, 0.3),
  c(0.5, 0.3, 1.0)
), byrow = TRUE, ncol = 3)
gen <- semPower.genSigma(Phi = Phi, nIndicator = c(3, 4, 5), 
                         loadM = c(.7, .6, .5))

# same as above, but using a factor loadings matrix
Lambda <- matrix(c(
  c(0.8, 0.0, 0.0),
  c(0.7, 0.0, 0.0),
  c(0.6, 0.0, 0.0),
  c(0.0, 0.7, 0.0),
  c(0.0, 0.8, 0.0),
  c(0.0, 0.5, 0.0),
  c(0.0, 0.4, 0.0),
  c(0.0, 0.0, 0.5),
  c(0.0, 0.0, 0.4),
  c(0.0, 0.0, 0.6),
  c(0.0, 0.0, 0.4),
  c(0.0, 0.0, 0.5)
), byrow = TRUE, ncol = 3)
gen <- semPower.genSigma(Phi = Phi, Lambda = Lambda)

# same as above, but using a reduced loading matrix, i. e.
# only define the primary loadings for each factor
loadings <- list(
  c(0.8, 0.7, 0.6),
  c(0.7, 0.8, 0.5, 0.4),
  c(0.5, 0.4, 0.6, 0.4, 0.5)
)
gen <- semPower.genSigma(Phi = Phi, loadings = loadings)

# Provide Beta for a three factor model
# with 3, 4, and 5 indicators 
# loading by .6, 5, and .4, respectively.
Beta <- matrix(c(
                c(0.0, 0.0, 0.0),
                c(0.3, 0.0, 0.0),  # f2 = .3*f1
                c(0.2, 0.4, 0.0)   # f3 = .2*f1 + .4*f2
               ), byrow = TRUE, ncol = 3)
gen <- semPower.genSigma(Beta = Beta, nIndicator = c(3, 4, 5), 
                         loadM = c(.6, .5, .4))

# two group example: 
# correlated two factor model (r = .25 and .35 in the first and second group, 
# respectively)
# the first factor is indicated by four indicators loading by .7 in the first 
# and .5 in the second group,
# the second factor is indicated by five indicators loading by .6 in the first 
# and .8 in the second group,
# all item intercepts are zero in both groups, 
# the latent means are zero in the first group
# and .25 and .10 in the second group.
gen <- semPower.genSigma(Phi = list(.25, .35), 
                         nIndicator = list(c(4, 5), c(4, 5)), 
                         loadM = list(c(.7, .6), c(.5, .8)), 
                         tau = list(rep(0, 9), rep(0, 9)), 
                         Alpha = list(c(0, 0), c(.25, .10))
                         )
gen[[1]]$Sigma  # implied covariance matrix group 1 
gen[[2]]$Sigma  # implied covariance matrix group 2
gen[[1]]$mu     # implied means group 1 
gen[[2]]$mu     # implied means group 2
}
}
