% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan_zip.R
\name{scan_zip}
\alias{scan_zip}
\title{Calculate the ZIP scan statistic.}
\usage{
scan_zip(table, zones, n_mcsim = 0, ...)
}
\arguments{
\item{table}{A \code{data.table} with columns 
\code{location, duration, count, mu, p}. The \code{location} column 
should consist of integers that are unique to each location. The 
\code{duration} column should also consist of integers, starting at 1 for 
the most recent time period and increasing in reverse chronological order.
The column \code{mu} should contain the estimated Poisson expected value 
parameters, and the column \code{p} the estimated structural zero 
probabilities.}

\item{zones}{A \code{set} of zones, each zone itself a set containing one or 
more locations of those found in \code{table}.}

\item{n_mcsim}{A non-negative integer; the number of replicate scan 
statistics to generate in order to calculate a p-value.}

\item{...}{Arguments passed to internal functions. Arguments that can be
passed here are \code{d}, the initial value for the excess zero indicators
(default is 0.5), and \code{tol}, the threshold for the absolute 
convergence criterion (default is 0.01).}
}
\value{
An object of class \code{scanstatistics}. It has the following 
   fields:
   \describe{
    \item{observed}{A \code{data.table} containing the value of the 
                    statistic calculated for each zone-duration combination,
                    for the observed data. The scan statistic is the maximum
                    value of these calculated statistics.}
    \item{replicated}{A numeric vector of length \code{n_mcsim} containing 
                      the values of the scanstatistics calculated by Monte
                      Carlo simulation.}
    \item{mlc}{A \code{data.table} containing the zone, duration, and 
               scanstatistic.}
    \item{pvalue}{The p-value calculated from Monte Carlo replications.}
    \item{distribution}{The assumed distribution of the data; "zero-inflated 
                        Poisson" in this case.}
    \item{type}{The type of scan statistic; "Expectation-based" in this 
                case.}
    \item{zones}{The set of zones that was passed to the function as input.}
    \item{n_locations}{The number of locations in the data.}
    \item{n_zones}{The number of zones.}
    \item{max_duration}{The maximum anomaly duration considered.}
   }
}
\description{
Calculate the expectation-based zero-inflated Poisson scan statistic by 
supplying a \code{data.table} of observed counts and pre-computed expected 
values and structural zero probabilities for each location and time. A 
p-value for the observed scan statistic can be obtained by Monte Carlo 
simulation.
}
\details{
For the expectation-based zero-inflated Poisson scan statistic 
   (Kjellson 2015), the null hypothesis of no anomaly holds that the count 
   observed at each location \eqn{i} and duration \eqn{t} (the number of time 
   periods before present) has a zero-inflated Poisson distribution with 
   expected value parameter \eqn{\mu_{it}} and structural zero probability
   \eqn{p_{it}}:
   \deqn{
     H_0 : Y_{it} \sim \textrm{ZIP}(\mu_{it}, p_{it}).
   }
   This holds for all locations \eqn{i = 1, \ldots, m} and all durations 
   \eqn{t = 1, \ldots,T}, with \eqn{T} being the maximum duration considered.
   Under the alternative hypothesis, there is a space-time window \eqn{W}
   consisting of a spatial zone \eqn{Z \subset \{1, \ldots, m\}} and a time 
   window \eqn{D \subseteq \{1, \ldots, T\}} such that the counts in that
   window have their Poisson expected value parameters inflated by a factor 
   \eqn{q_W > 1} compared to the null hypothesis:
   \deqn{
   H_1 : Y_{it} \sim \textrm{ZIP}(q_W \mu_{it}, p_{it}), ~~(i,t) \in W.
   }
   For locations and durations outside of this window, counts are assumed to
   be distributed as under the null hypothesis. The sets \eqn{Z} considered 
   are those specified in the argument \code{zones}, while the maximum 
   duration \eqn{T} is taken as the maximum value in the column 
   \code{duration} of the input \code{table}. 
   
   For each space-time window \eqn{W} considered, (the log of) a likelihood 
   ratio is computed using the distributions under the alternative and null 
   hypotheses, and the expectation-based Poisson scan statistic is calculated 
   as the maximum of these quantities over all space-time windows. The 
   expectation-maximization (EM) algorithm is used to obtain maximum 
   likelihood estimates. Point estimates of the parameters \eqn{\mu_{it}} 
   must be specified in the column \code{mu} of the argument \code{table} 
   before this function is called.
}
\examples{
# Simple example
set.seed(1)
table <- scanstatistics:::create_table(list(location = 1:4, duration = 1:4),
                                        keys = c("location", "duration"))
table[, mu := 3 * location]
table[, p := runif(.N, 0, 0.3)]
table[, count := gamlss.dist::rZIP(.N, mu = mu, sigma = p)]
table[location \%in\% c(1, 4) & duration < 3, 
      count := gamlss.dist::rZIP(.N, mu = 2 * mu, sigma = p)]
zones <- scanstatistics:::powerset_zones(4)
result <- scan_poisson(table, zones, 100)
result
}
\references{
Kjellson, B. (2015), \emph{Spatiotemporal Outbreak Detection: A Scan 
   Statistic Based on the Zero-Inflated Poisson Distribution}, (Master 
   Thesis, Stockholm University),
   \href{http://goo.gl/6Q89ML}{Link to PDF}.
}
\concept{
zero-inflated poisson scanstatistic
}

